package com.humandevice.android.appversioncontrol;

import android.app.Dialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.res.TypedArray;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.CallSuper;
import android.support.annotation.IntRange;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;
import android.support.annotation.StyleRes;
import android.support.v7.widget.Toolbar;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;

import java.io.Serializable;

/**
 * TODO: Dokumentacja
 *
 * @author Rafal Zajfert
 * @date 2016-09-08
 */
public class OldVersionDialog extends Dialog {

	private Toolbar mToolbar;
	private TextView mMessageTextView;
	private Button mPositiveButton;
	private Button mNegativeButton;

	@NonNull
	private Parameters mParameters;

	public OldVersionDialog(Context context) {
		super(context, R.style.AppVersionControlTheme);
		mParameters = new Parameters(context, R.style.AppVersionControlTheme);
	}

	public OldVersionDialog(Context context, int themeResId) {
		super(context, themeResId);
		mParameters = new Parameters(context, themeResId);
	}

	public OldVersionDialog(Context context, boolean cancelable, OnDialogCancelListener cancelListener) {
		super(context, R.style.AppVersionControlTheme);
		mParameters = new Parameters(context, R.style.AppVersionControlTheme);
		mParameters.mCancelable = cancelable;
		mParameters.mCancelListener = cancelListener;
	}

	private OldVersionDialog(@NonNull Parameters parameters) {
		super(parameters.mContext, parameters.mTheme);
		mParameters = parameters;
	}

	@Override
	@CallSuper
	protected void onCreate(Bundle savedInstanceState) {
		setContentView();

		mToolbar = (Toolbar) findViewById(R.id.toolbar);
		mMessageTextView = (TextView) findViewById(R.id.text);
		mPositiveButton = (Button) findViewById(R.id.positive_button);
		mNegativeButton = (Button) findViewById(R.id.negative_button);
		setupView();
	}

	private void setContentView() {
		if (mParameters.mContentView != null) {
			setContentView(mParameters.mContentView);
		} else {
			setContentView(mParameters.mContentViewRes);
		}
	}

	@Override
	public void setContentView(int layoutResID) {
		TypedArray typedArray = getContext().obtainStyledAttributes(mParameters.mTheme, R.styleable.AppVersionDialog);
		if (layoutResID == R.layout.dialog_old_version && !typedArray.hasValue(R.styleable.AppVersionDialog_backgroundColor)) {
			throw new RuntimeException("AppVersionDialog theme must contains backgroundColor parameter");
		}
		typedArray.recycle();
		super.setContentView(layoutResID);
	}

	private void setupView() {
		setCancelable(mParameters.mCancelable);
		setCanceledOnTouchOutside(mParameters.mCancelable);
		setOnCancelListener(mParameters.mCancelListener);

		CharSequence title = mParameters.mTitle != null ? mParameters.mTitle : getAppName();
		CharSequence message = mParameters.mMessage != null ? mParameters.mMessage : getContext().getString(R.string.too_old_app_short_message, getAppName());
		String positiveButtonText = mParameters.mPositiveButtonText != null ? mParameters.mPositiveButtonText : getContext().getString(R.string.go_to_google_play);
		String negativeButtonText = mParameters.mNegativeButtonText != null ? mParameters.mNegativeButtonText : getContext().getString(R.string.close);
		OnDialogClickListener positiveButtonListener = mParameters.mPositiveButtonListener;
		OnDialogClickListener negativeButtonListener = mParameters.mNegativeButtonListener;
		if (positiveButtonListener == null) {
			positiveButtonListener = new OnDialogClickListener() {
				@Override
				public void onClick(View v) {
					final String appPackageName = getContext().getPackageName();
					try {
						getContext().startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=" + appPackageName)));
					} catch (ActivityNotFoundException e) {
						getContext().startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/details?id=" + appPackageName)));
					}
				}
			};
		}
		if (negativeButtonListener == null) {
			negativeButtonListener = new OnDialogClickListener() {
				@Override
				public void onClick(View v) {
					if (mParameters.mCritical) {
						System.exit(0);
					} else {
						dismiss();
					}
				}
			};
		}


		setTitle(title);
		setMessage(message);
		setPositiveButton(positiveButtonText, positiveButtonListener);
		setNegativeButton(negativeButtonText, negativeButtonListener);
	}

	@Override
	public void setTitle(CharSequence title) {
		mParameters.mTitle = (String) title;
		if (mToolbar != null) {
			mToolbar.setTitle(title);
		}
	}

	@Override
	public void setTitle(@StringRes int titleId) {
		setTitle(getContext().getString(titleId));
	}

	public void setMessage(CharSequence message) {
		mParameters.mMessage = message;
		if (mMessageTextView != null) {
			mMessageTextView.setText(message);
		}
	}

	public void setMessage(@StringRes int messageId) {
		setMessage(getContext().getString(messageId));
	}

	private void setPositiveButton(String label, OnDialogClickListener listener) {
		mParameters.mPositiveButtonText = label;
		mParameters.mPositiveButtonListener = listener;
		if (mPositiveButton != null) {
			mPositiveButton.setText(label);
			mPositiveButton.setOnClickListener(listener);
		}
	}

	private void setPositiveButton(@StringRes int labelId, OnDialogClickListener listener) {
		setPositiveButton(getContext().getString(labelId), listener);
	}

	private void setNegativeButton(String label, OnDialogClickListener listener) {
		mParameters.mNegativeButtonText = label;
		mParameters.mNegativeButtonListener = listener;
		if (mNegativeButton != null) {
			mNegativeButton.setText(label);
			mNegativeButton.setOnClickListener(listener);
		}
	}

	private void setNegativeButton(@StringRes int labelId, OnDialogClickListener listener) {
		setNegativeButton(getContext().getString(labelId), listener);
	}

	private String getAppName() {
		int stringId = getContext().getApplicationInfo().labelRes;
		return getContext().getString(stringId);

	}

	private static class Parameters implements Serializable {

		transient private Context mContext;

		@StyleRes
		private int mTheme = R.style.AppVersionControlTheme;

		private boolean mCancelable;

		private OnDialogCancelListener mCancelListener;

		@LayoutRes
		private int mContentViewRes = R.layout.dialog_old_version;

		private View mContentView;

		private CharSequence mTitle;

		private CharSequence mMessage;

		private String mPositiveButtonText;

		private String mNegativeButtonText;

		private OnDialogClickListener mPositiveButtonListener;

		private OnDialogClickListener mNegativeButtonListener;

		private boolean mCritical;

		public Parameters(Context context) {
			mContext = context;
		}

		public Parameters(Context context, @StyleRes int theme) {
			mContext = context;
			mTheme = theme;
		}
	}

	public static class Builder implements Serializable {

		private Parameters mParameters;

		public Builder(Context context) {
			mParameters = new Parameters(context);
		}

		public Builder(Context context, @IntRange(from = 1) @StyleRes int theme) {
			mParameters = new Parameters(context, theme);
		}

		public Builder setContext(Context context) {
			mParameters.mContext = context;
			return this;
		}

		public Builder setTheme(@StyleRes int theme) {
			mParameters.mTheme = theme;
			return this;
		}

		public Builder setContentViewRes(int contentViewRes) {
			mParameters.mContentViewRes = contentViewRes;
			return this;
		}

		public Builder setContentView(View contentView) {
			mParameters.mContentView = contentView;
			return this;
		}

		public Builder setCancelable(boolean cancelable) {
			mParameters.mCancelable = cancelable;
			return this;
		}

		public Builder setOnCancelListener(OnDialogCancelListener cancelListener) {
			mParameters.mCancelListener = cancelListener;
			return this;
		}

		public Builder setTitle(CharSequence title) {
			mParameters.mTitle = title;
			return this;
		}

		public Builder setTitle(@StringRes int title) {
			mParameters.mTitle = mParameters.mContext.getString(title);
			return this;
		}

		public Builder setMessage(CharSequence message) {
			mParameters.mMessage = message;
			return this;
		}

		public Builder setMessage(@StringRes int message) {
			mParameters.mMessage = mParameters.mContext.getString(message);
			return this;
		}

		public Builder setPositiveButton(String label, @Nullable OnDialogClickListener listener) {
			mParameters.mPositiveButtonText = label;
			mParameters.mPositiveButtonListener = listener;
			return this;
		}

		public Builder setNegativeButtonText(String label, @Nullable OnDialogClickListener listener) {
			mParameters.mNegativeButtonText = label;
			mParameters.mNegativeButtonListener = listener;
			return this;
		}

		public Builder setCritical(boolean critical) {
			mParameters.mCritical = critical;
			return this;
		}

		public OldVersionDialog create() {
			return new OldVersionDialog(mParameters);
		}

		public OldVersionDialog show() {
			OldVersionDialog dialog = create();
			dialog.show();
			return dialog;
		}

	}

	private abstract class OnDialogClickListener implements View.OnClickListener, Serializable {
	}
	private abstract class OnDialogCancelListener implements OnCancelListener, Serializable {
	}
}
