package com.humandevice.android.database;

import android.database.sqlite.SQLiteDatabase;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.HashSet;
import java.util.Iterator;

/**
 * Sql table declaration helper.
 * @author Mikołaj Styś
 * @date 2015-09-05.
 */
public class SqlTableFactory {

    private final String tableName;
    private final HashSet<SqlColumn> columns;

    public SqlTableFactory(String tableName) {
        this.tableName = tableName;
        this.columns = new HashSet<>();
    }

    /**
     * Delete column by specified name. Null if not found. Convenient for changing reflect-made declaration.
     */
    public void deleteColumn(@NonNull String name) {
        Iterator<SqlColumn> columnIterator = columns.iterator();
        while (columnIterator.hasNext()) {
            SqlColumn column = columnIterator.next();
            if (name.equals(column.name)) {
                columnIterator.remove();
                return;
            }
        }
    }

    /**
     * Get column by specified name. Null if not found. Convenient for changing reflect-made declaration.
     */
    @Nullable
    public SqlColumn getColumn(@NonNull String name) {
        for (SqlColumn column : columns) {
            if (name.equals(column.name)) {
                return column;
            }
        }
        return null;
    }

    public SqlTableFactory addColumn(SqlColumn column) {
        columns.add(column);
        return this;
    }

    /**
     * Adds one column with specified columnName and type without any constrains
     */
    public SqlTableFactory addColumn(String columnName, SqlColumnType type) {
        columns.add(new SqlColumn(columnName, type));
        return this;
    }

    /**
     * @return Creates sql query for creating table
     */
    public String createQuery() {
        StringBuilder createQuery = new StringBuilder()
                .append("CREATE TABLE ").append(tableName).append(" (");
        boolean references = false;
        for (SqlColumn column : columns) {
            createQuery.append(column.name).append(' ').append(column.type.name());
            obtainModifiers(column, createQuery);
            createQuery.append(", ");
        }
        for (SqlColumn column : columns) {
            if (column.foreignTable != null) {
                createQuery.append("FOREIGN KEY(").append(column.name).append(") REFERENCES ")
                        .append(column.foreignTable).append("(").append(column.foreignColumn).append(")");
                createQuery.append(" ON UPDATE ").append(column.onUpdate.getSqlName());
                createQuery.append(" ON DELETE ").append(column.onDelete.getSqlName());
                createQuery.append(", ");
            }
        }
        createQuery.setLength(createQuery.length() - 2);
        createQuery.append(')');
        return createQuery.toString();
    }

    private void obtainModifiers(SqlColumn column, StringBuilder createQuery) {
        if (column.primaryKey) {
            createQuery.append(" PRIMARY KEY");
            if (column.autoincrement) {
                createQuery.append(" AUTOINCREMENT");
            }
        } else if (column.unique) {
            createQuery.append(" UNIQUE");
        } else if (column.notNull) {
            createQuery.append(" NOT NULL");
        }
    }

    /**
     * Executes generated code in order to create table
     *
     * @param db Dataabse to create table in
     */
    public void createTable(SQLiteDatabase db) {
        db.execSQL(createQuery());
    }
}
