package com.humandevice.android.database.dao;

import android.content.ContentValues;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.humandevice.android.database.SqlCursor;
import com.humandevice.android.database.tools.DatabaseAnnotationHelper;
import com.humandevice.android.database.tools.ReflectTypeUtils;

import java.lang.reflect.Field;
import java.util.Calendar;
import java.util.Date;

/**
 * Reflective implementation of {@link DatabaseTransformer}
 * @date 2016/01/27
 * @author Mikołaj Styś
 */
public class ReflectDatabaseTransformer<Model> implements DatabaseTransformer<Model> {

    private DatabaseAnnotationHelper helper = new DatabaseAnnotationHelper();
    private Class<Model> entityClass;

    public ReflectDatabaseTransformer() {

    }

    public ReflectDatabaseTransformer(Class<Model> entityClass) {
        this.entityClass = entityClass;
    }

    public void setEntityClass(Class<Model> entityClass) {
        this.entityClass = entityClass;
    }

    @Override
    public Model deserializeModel(SqlCursor.Row row, Model model) throws Exception {
        if (model == null) {
            model = entityClass.newInstance();
        }
        for (Field field : ReflectTypeUtils.getAllFields(entityClass)) {
            String databaseName = helper.getDatabaseName(field);
            if (!TextUtils.isEmpty(databaseName)) {
                bindValues(databaseName, field, model, row);
            }
        }
        return model;
    }

    private void bindValues(String databaseName, Field field, Model model, SqlCursor.Row row)
            throws IllegalAccessException, NoSuchFieldException {
        if (!row.hasValueFor(databaseName)) {
            return;
        }
        Class type = field.getType();
        field.setAccessible(true);
        if (ReflectTypeUtils.isLong(type)) {
            field.setLong(model, row.getLong(databaseName));
        } else if (ReflectTypeUtils.isString(type)) {
            field.set(model, row.getString(databaseName));
        } else if (ReflectTypeUtils.isFixedPrecision(type)) {
            field.setInt(model, row.getInteger(databaseName));
        } else if (ReflectTypeUtils.isBoolean(type)) {
            field.setBoolean(model, row.getInteger(databaseName) == 1);
        } else if (ReflectTypeUtils.isFloatingPoint(type)) {
            field.setDouble(model, row.getDouble(databaseName));
        } else if (Calendar.class.equals(type)) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis(row.getLong(databaseName));
            field.set(model, calendar);
        } else if (ReflectTypeUtils.isDate(type)) {
            field.set(model, new Date(row.getLong(databaseName)));
        } else if (ReflectTypeUtils.isEnum(type)) {
            field.set(model, helper.getEnumFromDatabaseName(field, row.getString(databaseName)));
        } else {
            throw new IllegalStateException("Cannot bind to field for " + field.getName() +
                    " with name: " + databaseName);
        }
        field.setAccessible(false);
    }

    @Override
    public ContentValues serializeModel(Model model, @Nullable ContentValues form) throws Exception {
        if (form == null) {
            form = new ContentValues();
        }
        for (Field field : ReflectTypeUtils.getAllFields(entityClass)) {
            String databaseName = helper.getDatabaseName(field);
            if (!TextUtils.isEmpty(databaseName)) {
                putValue(model, field, databaseName, form);
            }
        }
        return form;
    }

    private void putValue(Model model, Field field, String databaseName, ContentValues values) throws IllegalAccessException, NoSuchFieldException {
        Class type = field.getType();
        field.setAccessible(true);
        if (field.get(model) == null) {
            field.setAccessible(false);
            return;
        }
        if (ReflectTypeUtils.isLong(type)) {
            values.put(databaseName, field.getLong(model));
        } else if (ReflectTypeUtils.isFixedPrecision(type)) {
            values.put(databaseName, field.getInt(model));
        } else if (ReflectTypeUtils.isBoolean(type)) {
            if (field.getBoolean(model)) {
                values.put(databaseName, 1);
            } else {
                values.put(databaseName, 0);
            }
        } else if (ReflectTypeUtils.isFloatingPoint(type)) {
            values.put(databaseName, field.getDouble(model));
        } else if (ReflectTypeUtils.isEnum(type)) {
            values.put(databaseName, helper.getEnumDatabaseName(field, model));
        } else if (Calendar.class.equals(type)) {
            if (field.get(model) == null) {
                values.put(databaseName, 0L);
            } else {
                values.put(databaseName, ((Calendar) field.get(model)).getTimeInMillis());
            }
        } else if (ReflectTypeUtils.isDate(type)) {
            if (field.get(model) == null) {
                values.put(databaseName, 0L);
            } else {
                values.put(databaseName, ((Date) field.get(model)).getTime());
            }
        } else if (ReflectTypeUtils.isString(type)) {
            values.put(databaseName, field.get(model).toString());
        } else if (ReflectTypeUtils.isByteArray(type)) {
            values.put(databaseName, (byte[]) field.get(model));
        } else {
            throw new IllegalAccessException("Cannot extract simple type from field: " + field.getName()
                    + " and database name: " + databaseName + " type is: " + type);
        }
        field.setAccessible(false);
    }
}
