package com.humandevice.android.database;

import android.database.Cursor;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.io.Closeable;
import java.util.Iterator;

/**
 * Wraps android native cursor to provide some additional functions like iteration.
 * @author Mikołaj Styś
 * @date 2015-09-05
 */
public class SqlCursor implements Iterable<SqlCursor.Row>, Iterator<SqlCursor.Row>, Closeable {

    @Nullable
    protected Cursor mCursor;

    public SqlCursor(@Nullable Cursor cursor) {
        this.mCursor = cursor;
    }

    public SqlCursor() {
        this(null);
    }

    public void setNativeCursor(@Nullable Cursor cursor) {
        this.mCursor = cursor;
    }

    @Override
    public boolean hasNext() {
        return hasNextRow();
    }

    /**
     * Moves cursor to next row
     */
    @Override
    public Row next() {
        moveToNextRow();
        return getRow();
    }

    @Override
    /**
     * Not implemented. Do not use!
     */
    public void remove() {
        throw new UnsupportedOperationException("If you want to delete, call dao's method!");
    }

    private boolean hasNextRow() {
        return mCursor != null && mCursor.getPosition() + 1 < mCursor.getCount();
    }

    private boolean moveToNextRow() {
        return mCursor != null && mCursor.moveToNext();
    }

    @Override
    public void close() {
        if (mCursor != null) {
            mCursor.close();
        }
        mCursor = null;
    }


    public boolean hasRows() {
        return mCursor != null && mCursor.getCount() > 0;
    }

    /**
     * Get cursor size. 0 if cursor is empty or not exists.
     */
    public int getSize() {
        if (mCursor == null) {
            return 0;
        }
        return mCursor.getCount();
    }

    @Nullable
    /**
     * Get Row object that blocks moving cursor to another row;
     */
    public Row getRow() {
        if (mCursor != null) {
            if (mCursor.isBeforeFirst()) {
                mCursor.moveToFirst();
            } else if (mCursor.isAfterLast()) {
                return null;
            }
            return new Row(mCursor);
        } else {
            return null;
        }
    }

    @Override
    public Iterator<Row> iterator() {
        return this;
    }

    /**
     * Represents on row of cursor's data
     */
    public class Row {

        @NonNull
        protected Cursor cursor;

        public boolean hasValueFor(String name) {
            return hasValueFor(cursor.getColumnIndex(name));
        }

        public boolean hasValueFor(int columnIndex) {
            return !cursor.isNull(columnIndex);
        }

        public Row(@NonNull Cursor cursor) {
            this.cursor = cursor;
        }

        public long getLong(int column) {
            return cursor.getLong(column);
        }

        public int getSize() {
            return cursor.getColumnCount();
        }

        public String getString(String columnName) {
            return cursor.getString(cursor.getColumnIndex(columnName));
        }

        public long getLong(String columnName) {
            return getLong(cursor.getColumnIndex(columnName));
        }

        public int getInteger(String columnName) {
            return getInteger(cursor.getColumnIndex(columnName));
        }

        private int getInteger(int columnIndex) {
            return cursor.getInt(columnIndex);
        }

        public double getDouble(String columnName) {
            return getDouble(cursor.getColumnIndex(columnName));
        }

        public double getDouble(int columnIndex) {
            return cursor.getDouble(columnIndex);
        }

        public float getFloat(String columnName) {
            return getFloat(cursor.getColumnIndex(columnName));
        }

        public float getFloat(int columnIndex) {
            return cursor.getFloat(columnIndex);
        }
    }
}
