package com.humandevice.android.database.dao;

import android.app.Application;
import android.content.ContentValues;
import android.database.sqlite.SQLiteDatabase;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.humandevice.android.core.model.Identifiable;
import com.humandevice.android.core.view.annotation.AppContext;
import com.humandevice.android.database.DatabaseConfiguration;
import com.humandevice.android.database.DatabaseHelper;
import com.humandevice.android.database.SqlCursor;
import com.humandevice.android.database.model.LocalIdentifiable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

/**
 * Provides basic operations to save / restore object data in database.
 * Some operations like {@link #update(Object)} needs implemented {@link LocalIdentifiable}
 * for serialized object
 *
 * @author Mikołaj Styś
 * @date 01/06/2016
 */
public abstract class AbstractDao<DbObject> {

    public static final String ID = "_id";

    protected static DatabaseHelper sDatabaseHelper;

    protected DatabaseTransformer<DbObject> mTransformer;
    private SQLiteDatabase mDatabase;
    
    public static void setDatabaseConfiguration(@AppContext Application context, DatabaseConfiguration configuration) {
        sDatabaseHelper = new DatabaseHelper(context, configuration);
    }

    public static void closeDatabase() {
        if (sDatabaseHelper != null) {
            sDatabaseHelper.close();
        }
        sDatabaseHelper = null;
    }

    protected AbstractDao(DatabaseTransformer<DbObject> transformer) {
        if (sDatabaseHelper == null) {
            throw new IllegalStateException("call setDatabaseConfiguration in Application class first!");
        }
        this.mTransformer = transformer;
    }

    @NonNull
    public abstract String getTableName();

    /**
     * Method is used to create necessary tables for this dao.
     */
    public abstract void onTableCreate(@NonNull SQLiteDatabase db);

    /**
     * Invoked after all {@link #onTableCreate(SQLiteDatabase)} methods are invoked
     */
    public void onPostCreate(@NonNull SQLiteDatabase db) {

    }

    /**
     * Returns entity that contains this key
     *
     * @param entityId key to find entity
     * @return entity or null if Object do not have id column
     */
    @Nullable
    public DbObject getById(final long entityId) {
        return getOneResult(ID + " = ?", Long.toString(entityId));
    }

    public boolean add(@NonNull final DbObject entity) {
        SQLiteDatabase database = getWritableDatabase();
        return insert(entity, database, SQLiteDatabase.CONFLICT_NONE);
    }

    public boolean add(@NonNull final DbObject[] entities) {
        return add(Arrays.asList(entities));
    }

    public boolean add(@NonNull final Collection<DbObject> entities) {
        SQLiteDatabase database = getWritableDatabase();
        database.beginTransaction();
        boolean success = true;
        for (DbObject entity : entities) {
            success = insert(entity, database, SQLiteDatabase.CONFLICT_NONE) && success;
        }
        database.setTransactionSuccessful();
        database.endTransaction();
        return success;
    }

    public boolean addOrReplace(@NonNull DbObject entity) {
        SQLiteDatabase database = getWritableDatabase();
        return insert(entity, database, SQLiteDatabase.CONFLICT_REPLACE);
    }

    public boolean addOrReplace(@NonNull DbObject[] entities) {
        return addOrReplace(Arrays.asList(entities));
    }

    public boolean addOrReplace(@NonNull Collection<DbObject> entities) {
        SQLiteDatabase database = getWritableDatabase();
        database.beginTransaction();
        boolean success = true;
        for (DbObject entity : entities) {
            success = insert(entity, database, SQLiteDatabase.CONFLICT_REPLACE) && success;
        }
        database.setTransactionSuccessful();
        database.endTransaction();
        return success;
    }

    @Nullable
    protected ContentValues prepareValues(@NonNull DbObject entity) {
        ContentValues values = null;
        try {
            values = mTransformer.serializeModel(entity, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return values;
    }

    /**
     * Prepares values using transformer. Internal method for inserting.
     * If id is valid - uses it for insert.
     * @return true if something was inserted
     */
    protected boolean insert(@NonNull DbObject entity, @NonNull SQLiteDatabase database, int conflictAlgorithm) {
        ContentValues values = prepareValues(entity);
        if (values == null) {
            Log.w(this.getClass().getSimpleName(), "No values to insert");
            return false;
        }
        if (entity instanceof LocalIdentifiable) {
            Object id = values.get(ID);
            if (id != null) {
                if (id.equals(0)) {
                    Log.w(this.getClass().getSimpleName(), "Inserting entity with id = 0, maybe you forget about setting id = INVALID_ID?");
                } else if (id.equals(Identifiable.INVALID_ID)) {
                    values.remove(ID);
                }
            }
            long result = database.insertWithOnConflict(getTableName(), null, values, conflictAlgorithm);
            if (result > -1) {
                ((LocalIdentifiable) entity).setDatabaseId(result);
                return true;
            }
            return false;
        } else {
            return database.insertWithOnConflict(getTableName(), null, values, conflictAlgorithm) > -1;
        }
    }


    /**
     * Updates entity in database
     *
     * @param entity objects that implements LocalIdentifiable
     * @return true if entity was updated
     */
    public boolean update(@NonNull final DbObject entity) {
        if (entity instanceof LocalIdentifiable) {
            ContentValues values = prepareValues(entity);
            if (values == null) {
                Log.w(this.getClass().getSimpleName(), "No values to update");
                return false;
            }
            return update(values, ID + " = ?", Long.toString(((LocalIdentifiable) entity).getDatabaseId())) > 0;
        } else {
            throw new UnsupportedOperationException("Override update function to update objects " +
                    "that do not implement LocalIdentifiable interface!");
        }
    }

    /**
     * @return the number of rows affected
     */
    protected int update(@NonNull ContentValues values, @Nullable final String whereClause, @Nullable final String... arguments) {
        SQLiteDatabase database = getWritableDatabase();
        return database.update(getTableName(), values, whereClause, arguments);
    }


    /**
     * Removes entity from database if implements LocalIdentifiable
     *
     * @param entity objects that implements LocalIdentifiable
     * @return true if something was deleted
     */
    public boolean delete(final DbObject entity) {
        if (entity instanceof LocalIdentifiable) {
            boolean result;
            result = delete(ID + " = ?", Long.toString(((LocalIdentifiable) entity).getDatabaseId())) > 0;
            ((LocalIdentifiable) entity).setDatabaseId(LocalIdentifiable.INVALID_ID);
            return result;
        } else {
            throw new UnsupportedOperationException("Override delete function to delete objects " +
                    "that do not implement LocalIdentifiable interface!");
        }
    }

    /**
     * @return the number of rows affected if a whereClause is passed in, 0 otherwise.
     * To remove all rows and get a count pass "1" as the whereClause
     */
    protected int delete(@Nullable final String whereClause, @Nullable final String... arguments) {
        SQLiteDatabase database = getWritableDatabase();
        return database.delete(getTableName(), whereClause, arguments);
    }

    /**
     * @return all results in table
     */
    @NonNull
    public List<DbObject> getAll() {
        return getList(null);
    }

    /**
     * Clears table
     *
     * @return Number of deleted rows
     */
    public int deleteAll() {
        return delete("1");
    }

    /**
     * Counts rows
     *
     * @return the number of rows returned
     */
    public long count(@Nullable String whereClause, @Nullable final String... parameters) {
        SQLiteDatabase database = getReadableDatabase();
        SqlCursor cursor = new SqlCursor(
                database.rawQuery("SELECT COUNT(*) FROM " + getTableName() + (!TextUtils.isEmpty(whereClause) ? " WHERE " + whereClause : "") + ";", parameters));
        //noinspection ConstantConditions
        long count = cursor.getRow().getLong(0);
        cursor.close();
        return count;
    }

    /**
     * Applies to ALL database operations - invariant to used DAO
     */
    public void startTransaction() {
        getWritableDatabase().beginTransaction();
    }

    /**
     * Cancels transaction
     */
    public void cancelTransaction() {
        SQLiteDatabase db = getWritableDatabase();
        db.endTransaction();
    }

    /**
     * If {@link #cancelTransaction()} wasn't called, it will succeed
     */
    public void endTransaction() {
        SQLiteDatabase db = getWritableDatabase();
        db.setTransactionSuccessful();
        db.endTransaction();
    }

    /**
     * Abstract one result query
     *
     * @param selection          WHERE clause excluding WHERE keyword
     * @param selectionArguments parameters to be bind
     * @return list with result
     */
    @Nullable
    protected DbObject getOneResult(@Nullable final String selection, @Nullable final String... selectionArguments) {
        DbObject result = null;
        SQLiteDatabase database = getReadableDatabase();
        SqlCursor dbResult = new SqlCursor();
        //noinspection TryFinallyCanBeTryWithResources
        try {
            dbResult.setNativeCursor(database.query(getTableName(), null, selection, selectionArguments, null, null, null, "1"));
            if (dbResult.hasRows()) {
                result = mTransformer.deserializeModel(dbResult.getRow(), null);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            dbResult.close();
        }
        return result;
    }

    /**
     * Abstract list query that helps create queries
     *
     * @param selection          WHERE clause excluding WHERE keyword. Pass null for all rows
     * @param selectionArguments parameters to be bound
     * @return list with result
     */
    @NonNull
    protected List<DbObject> getList(@Nullable final String selection, @Nullable final String... selectionArguments) {
        List<DbObject> result = new LinkedList<>();
        SQLiteDatabase database = getReadableDatabase();
        SqlCursor dbResult = new SqlCursor();
        //noinspection TryFinallyCanBeTryWithResources
        try {
            dbResult.setNativeCursor(database.query(getTableName(), null, selection, selectionArguments, null, null, null));
            if (dbResult.hasRows()) {
                result = convertCursor(dbResult);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            dbResult.close();
        }
        return result;
    }

    /**
     * Converts multiple results into list
     */
    @NonNull
    protected List<DbObject> convertCursor(final SqlCursor dbResult) {
        List<DbObject> result = new ArrayList<>();
        try {
            for (SqlCursor.Row row : dbResult) {
                result.add(mTransformer.deserializeModel(row, null));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }
    
    protected SQLiteDatabase getReadableDatabase(){
        if (mDatabase!=null){
            return mDatabase;
        }else{
            return sDatabaseHelper .getReadableDatabase();
        }
    }

    protected SQLiteDatabase getWritableDatabase(){
        if (mDatabase!=null){
            return mDatabase;
        }else{
            return sDatabaseHelper .getWritableDatabase();
        }
    }
    
    public void setDatabase(SQLiteDatabase database){
        mDatabase = database;
    }
}