package com.humandevice.android.database.dao;

import android.database.sqlite.SQLiteDatabase;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.humandevice.android.database.SqlColumn;
import com.humandevice.android.database.SqlColumnType;
import com.humandevice.android.database.SqlTableFactory;
import com.humandevice.android.database.annotations.Table;
import com.humandevice.android.database.tools.DatabaseAnnotationHelper;
import com.humandevice.android.database.tools.ReflectTypeUtils;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

/**
 * Automatically converts annotated Entity to SQL values and provides table name.
 * Reflect version of {@link AbstractDao}
 *
 * @author Mikołaj Styś
 * @date 01/02/2016.
 * <br/>See:<br/>
 * {@link Table} <br/>
 * {@link com.humandevice.android.core.adapters.Serialized}
 */
public class ReflectDao<DbObject> extends AbstractDao<DbObject> {

    private final String mTableName;
    private Class<DbObject> mClass;

    /**
     * Use this constructor if ReflectDao is used for extended class.
     */
    protected ReflectDao() {
        this(null);
    }

    /**
     * Use this constructor if you directly instantiates ReflectDao.
     */
    @SuppressWarnings ("unchecked")
    public ReflectDao(Class<DbObject> cls) {
        super(new ReflectDatabaseTransformer<DbObject>());
        mClass = cls;
        if (mClass == null) {
            mClass = getParameterClass();
        }
        ((ReflectDatabaseTransformer) mTransformer).setEntityClass(mClass);
        Table tableAnnotation = mClass.getAnnotation(Table.class);
        if (tableAnnotation == null) {
            throw new IllegalStateException(mClass.getName() + " should be annotated using @Table annotation!");
        }
        mTableName = mClass.getAnnotation(Table.class).value();
    }

    /**
     * Obtains typed class if this class is subtyped.
     */
    @SuppressWarnings ("unchecked")
    protected Class<DbObject> getParameterClass() {
        ParameterizedType genericSuperclass = (ParameterizedType) getClass().getGenericSuperclass();
        Type type = genericSuperclass.getActualTypeArguments()[0];
        if (type instanceof Class) {
            return (Class<DbObject>) type;
        } else if (type instanceof ParameterizedType) {
            return (Class<DbObject>) ((ParameterizedType) type).getRawType();
        }
        return null;
    }

    /**
     * Override, if you want to change table name
     */
    @NonNull
    @Override
    public String getTableName() {
        return mTableName;
    }

    /**
     * Override if you need to manually change table declaration
     *
     * @return SqlTableFactory ready to be executed
     */
    protected SqlTableFactory getTableDeclaration() {
        SqlTableFactory tableFactory = new SqlTableFactory(getTableName());
        DatabaseAnnotationHelper helper = new DatabaseAnnotationHelper();
        for (Field field : ReflectTypeUtils.getAllFields(mClass)) {
            String name = helper.getDatabaseName(field);
            if (TextUtils.isEmpty(name)) {
                continue;
            }
            Class<?> type = field.getType();
            if (name.equals(ID)) {
                tableFactory.addColumn(new SqlColumn(ID, SqlColumnType.INTEGER).asId(helper.isAutoincrement(field)));
            } else if (ReflectTypeUtils.isFixedPrecision(type)) {
                tableFactory.addColumn(name, SqlColumnType.INTEGER);
            } else if (ReflectTypeUtils.isFloatingPoint(type)) {
                tableFactory.addColumn(name, SqlColumnType.REAL);
            } else if (ReflectTypeUtils.isDate(type)) {
                tableFactory.addColumn(name, SqlColumnType.INTEGER);
            } else if (ReflectTypeUtils.isBoolean(type)) {
                tableFactory.addColumn(name, SqlColumnType.INTEGER);
            } else if (ReflectTypeUtils.isString(type) || ReflectTypeUtils.isEnum(type)) {
                tableFactory.addColumn(name, SqlColumnType.TEXT);
            } else if (ReflectTypeUtils.isByteArray(type)) {
                tableFactory.addColumn(name, SqlColumnType.BLOB);
            } else {
                throw new IllegalStateException("Cannot find suitable type for column " + name + " and type " + type);
            }
        }
        return tableFactory;
    }

    @Override
    public void onTableCreate(@NonNull SQLiteDatabase db) {
        getTableDeclaration().createTable(db);
    }
}
