package com.humandevice.android.database;

import android.database.Cursor;

/**
 * Simple class containing column info.
 * @author Mikołaj Styś
 * @date 2015-09-05.
 */
public final class SqlColumn {

    public String name;
    public SqlColumnType type;

    boolean notNull;
    boolean primaryKey;
    boolean autoincrement;
    boolean unique;

    String foreignColumn;
    String foreignTable;
    SqlForeignKeyActions onDelete;
    SqlForeignKeyActions onUpdate;

    public SqlColumn() {
        type = SqlColumnType.NONE;
        name = "";
    }

    public SqlColumn(String columnName, SqlColumnType type) {
        this.name = columnName;
        this.type = type;
    }

    /**
     * Constructor does not close cursor nor changes its state
     *
     * @param cursor for creating cursor from table info
     */
    public SqlColumn(Cursor cursor) {
        String type = cursor.getString(cursor.getColumnIndex("type"));
        name = cursor.getString(cursor.getColumnIndex("name"));
        this.type = SqlColumnType.valueOf(type);
    }

    public SqlColumn setAutoincrement(boolean autoincrement) {
        this.autoincrement = autoincrement;
        return this;
    }

    public SqlColumn setPrimaryKey(boolean primaryKey) {
        this.primaryKey = primaryKey;
        return this;
    }

    @Override
    public boolean equals(Object o) {
        return o instanceof SqlColumn && ((SqlColumn) o).name.equalsIgnoreCase(this.name);
    }

    public SqlColumn setNotNull(boolean notNull) {
        this.notNull = notNull;
        return this;
    }

    public SqlColumn primaryKey() {
        this.primaryKey = true;
        return this;
    }

    public SqlColumn setUnique(boolean unique) {
        this.unique = unique;
        return this;
    }

    /**
     * Makes column unique
     */
    public SqlColumn unique() {
        setUnique(true);
        return this;
    }

    public SqlColumn asId(boolean isAutoincrement) {
        primaryKey();
        setAutoincrement(isAutoincrement);
        return this;
    }

    /**
     * @see #foreignKey(String, String, SqlForeignKeyActions, SqlForeignKeyActions)
     */
    public SqlColumn foreignKey(String tableName, String column) {
        foreignKey(tableName, column, SqlForeignKeyActions.NO_ACTION, SqlForeignKeyActions.NO_ACTION);
        return this;
    }

    /**
     * Create relation to specified column in specified table. Also it stats onDelete and onUpdate actions
     */
    public SqlColumn foreignKey(String tableName, String column, SqlForeignKeyActions onDeleteAction, SqlForeignKeyActions onUpdateAction) {
        foreignColumn = column;
        foreignTable = tableName;
        onDelete = onDeleteAction;
        onUpdate = onUpdateAction;
        return this;
    }
}
