package com.humandevice.android.formvalidator;

import android.support.annotation.NonNull;
import android.view.View;
import android.widget.TextView;


import com.humandevice.android.formvalidator.errors.FormErrors;
import com.humandevice.android.formvalidator.errors.ViewFormErrorException;
import com.humandevice.android.formvalidator.rules.Rule;
import com.humandevice.android.formvalidator.utils.ErrorTextWatcher;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

/**
 * Created by Rafal Zajfert on 10.08.2015.
 */
@SuppressWarnings("unused")
public class FormValidator {

    public static final int PRIORITY_ALL = Integer.MAX_VALUE;
    protected final Map<View, Set<Rule>> validators = new LinkedHashMap<>();
    protected final FormErrors errors = new FormErrors();

    public FormValidator() {
    }

    public void addRule(@NonNull View view, @NonNull Rule rule) {
        addRule(view, rule, new Rule[]{});
    }

    public void addRule(@NonNull View view, @NonNull Rule rule, Rule... rules) {
        Set<Rule> ruleList = getRuleList(view);

        if (view.getTag() == null){
            throw new IllegalArgumentException("View must have tag");
        }
        ruleList.add(rule);
        Collections.addAll(ruleList, rules);

        init(view);
    }

    protected void init(@NonNull View view) {
        if (view instanceof TextView) {
            initTextWatcher((TextView) view);
        }
    }

    @NonNull
    protected Set<Rule> getRuleList(@NonNull View view) {
        Set<Rule> ruleList;
        if (validators.containsKey(view)) {
            ruleList = validators.get(view);
        } else {
            ruleList = new TreeSet<>(new Rule.PriorityComparator());
            validators.put(view, ruleList);
        }
        return ruleList;
    }

    public void validate() {
        validate(PRIORITY_ALL);
    }

    public boolean hasErrors() {
        return errors.hasErrors();
    }

    public boolean hasErrors(View view) {
        return errors.hasErrors((String) view.getTag());
    }

    public FormErrors getErrors() {
        return errors;
    }

    public void validate(int maxPriority) {
        errors.clear();
        validateAllViews(maxPriority);
    }

    protected void validateAllViews(int maxPriority) {
        for (View view : validators.keySet()) {
            validateView(view, maxPriority);
        }
    }

    protected void validateView(@NonNull View view, int maxPriority) {
        Set<Rule> ruleList = validators.get(view);
        int errorRulePriority = Integer.MIN_VALUE;
        for (Rule rule : ruleList) {

            if (rule.getPriority() > maxPriority) {
                break;
            }
            if (errorRulePriority != Integer.MIN_VALUE && errorRulePriority != rule.getPriority()) {
                break;
            }

            try {
                rule.validate(view);
            } catch (ViewFormErrorException e) {
                errors.add(e.getFormError());
                errorRulePriority = rule.getPriority();
            }
        }
    }

    public void setErrors(@NonNull FormErrors errors){
        this.errors.clear();
        this.errors.addAll(errors);

    }

    protected void initTextWatcher(@NonNull TextView view) {
        ErrorTextWatcher textWatcher = new ErrorTextWatcher(view);
        view.removeTextChangedListener(textWatcher); //remove old textWatcher if was previously added
        view.addTextChangedListener(textWatcher);
    }
}
