package com.humandevice.android.formvalidator.errors;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by Rafal Zajfert on 10.08.2015.
 */
public class FormErrors {

    private Map<String, List<FormError>> errors = new LinkedHashMap<>();

    /**
     * @return List of all errors
     */
    @NonNull
    public List<FormError> getAllErrors() {
        List<FormError> list = new ArrayList<>();
        for (List<FormError> errorList : errors.values()) {
            list.addAll(errorList);
        }
        return list;
    }

    /**
     * Add new {@link FormError} to the list
     *
     * @param error
     */
    public void add(@NonNull FormError error) {
        List<FormError> ruleList = getErrorsList(error.getFieldTag());
        ruleList.add(error);
    }

    public void addAll(FormErrors errors) {
        for (FormError error : errors.getAllErrors()) {
            add(error);
        }
    }

    public Set<String> getFieldTagsWithError(){
        return errors.keySet();
    }

    /**
     * Returns errors list assigned to specified field
     *
     * @param fieldTag
     * @return
     */
    @NonNull
    private List<FormError> getErrorsList(@Nullable String fieldTag) {
        List<FormError> list;
        if (errors.containsKey(fieldTag)) {
            list = errors.get(fieldTag);
        } else {
            list = new ArrayList<>();
            errors.put(fieldTag, list);
        }
        return list;
    }

    /**
     * Remove all errors
     */
    public void clear() {
        errors.clear();
    }

    /**
     * Removes all errors for the specified field
     */
    public void remove(String fieldTag){
        if (errors.containsKey(fieldTag)) {
            errors.remove(fieldTag);
        }
    }

    /**
     * Check if the error list isEmpty
     */
    public boolean hasErrors(String fieldTag) {
        return errors.containsKey(fieldTag);
    }

    /**
     * Check if the error list isEmpty
     */
    public boolean hasErrors() {
        return !errors.isEmpty();
    }

    /**
     * Returns concatenated errors messages for specified tag. Each error is separated with new line
     */
    @NonNull
    public String getCombinedMessage(String fieldTag) {
        StringBuilder builder = new StringBuilder();
        for (FormError error : errors.get(fieldTag)) {
            if ((error.getFieldTag() == null && fieldTag == null) || error.getFieldTag() != null && error.getFieldTag().equals(fieldTag)) {
                if (builder.length() > 0) {
                    builder.append("\n");
                }
                builder.append(error.getMessage());
            }
        }
        return builder.toString();
    }

    /**
     * @return Errors count for all views
     */
    public int count() {
        int count = 0;
        for (List<FormError> errorList : errors.values()) {
            count += errorList.size();
        }
        return count;
    }


}
