package com.humandevice.maskwatcher;

import android.text.Editable;
import android.text.InputType;
import android.text.Selection;
import android.text.SpanWatcher;
import android.text.Spannable;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.widget.EditText;

/**
 * @author Rafal Zajfert
 */
public class InputMask implements TextWatcher {

	private final EditText mEditText;
	private MaskFormatter mMaskFormatter;

	private SpanWatcher mSpanWatcher = new SpanWatcher() {
		@Override
		public void onSpanAdded(final Spannable text, final Object what,
								final int start, final int end) {
		}

		@Override
		public void onSpanRemoved(final Spannable text, final Object what,
								  final int start, final int end) {
		}

		@Override
		public void onSpanChanged(final Spannable text, final Object what,
								  final int ostart, final int oend, final int nstart, final int nend) {
			if (what == Selection.SELECTION_START) {
				mMaskFormatter.changeSelection(nstart);
			}
		}
	};

	public InputMask(EditText editText, String mask) {
		mEditText = editText;
		mMaskFormatter = new MaskFormatter(mask);
	}

	public static InputMask applyTo(EditText editText, String mask) {
		InputMask watcher = new InputMask(editText, mask);
		editText.addTextChangedListener(watcher);
		editText.setInputType(InputType.TYPE_TEXT_FLAG_NO_SUGGESTIONS);
		return watcher;
	}


	@Override
	public void beforeTextChanged(CharSequence s, int start, int count, int after) {
		mEditText.getText().removeSpan(mSpanWatcher);
	}

	@Override
	public void onTextChanged(CharSequence s, int start, int before, int count) {
		if (mMaskFormatter.isEmptyMask()) {
			return;
		}
		mMaskFormatter.changeValue(s, start, before, count);
	}

	@Override
	public void afterTextChanged(Editable s) {
		if (mMaskFormatter.isEmptyMask()) {
			return;
		}
		CharSequence formattedString = mMaskFormatter.getText(false);

		if (!TextUtils.equals(s, formattedString)) {
			mEditText.removeTextChangedListener(this);
			s.replace(0, s.length(), formattedString);
			mEditText.addTextChangedListener(this);
		}

		mEditText.getText().setSpan(mSpanWatcher, 0, formattedString.length(), Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
		int selection = mMaskFormatter.getSelection();
		mEditText.setSelection(Math.max(0, Math.min(selection, formattedString.length())));
	}

	public void getText(boolean onlyValuable){
		mMaskFormatter.getText(onlyValuable);
	}
}
