package com.humandevice.maskwatcher;


import java.util.ArrayList;
import java.util.List;

/**
 * @author Rafal Zajfert
 */
class MaskFormatter {

	private final List<Char> mMask;
	private int mSelection;

	MaskFormatter(String mask) {
		mMask = processMask(mask);
	}

	private List<Char> processMask(String mask) {
		List<Char> maskChars = new ArrayList<>();
		char[] chars = mask.toCharArray();
		boolean escape = false;
		boolean valuable = false;
		Boolean uppercase = null;
		for (char c : chars) {
			if (!escape && c == '\\') {
				escape = true;
			} else if (!escape && (c == '(' || c == ')')) {
				valuable = c == '(';
			} else if (escape) {
				if (c == 'L') {
					uppercase = false;
				} else if (c == 'U') {
					uppercase = true;
				} else if (c == 'E') {
					uppercase = null;
				} else {
					maskChars.add(new Char.Builder().setValue(c).setValuable(valuable).setCase(uppercase).build());
				}
				escape = false;
			} else {
				maskChars.add(new Char.Builder(c).setValuable(valuable).setCase(uppercase).build());
			}
		}
		return maskChars;
	}

	boolean isEmptyMask() {
		return mMask.isEmpty();
	}


	void changeValue(CharSequence s, int start, int toDelete, int toAdd) {
		if (toDelete > 0) {
			if (toDelete == 1) {
				int initStart = start;
				while (start > 0 && mMask.get(start) instanceof Const) {
					toDelete++;
					start--;
				}
				if (start != initStart) {
					s = s.subSequence(0, start).toString() + s.subSequence(start + toDelete - 1, s.length());
				}
			}
			for (int i = start; i < start + toDelete; i++) {
				mMask.get(i).removeValue();
			}
		}

		int cursorPos = start + toAdd;

		int sequencePos = 0;
		int maskPos = 0;
		for (; maskPos < mMask.size(); maskPos++) {
			Char aChar = mMask.get(maskPos);
			while (sequencePos < s.length()) {
				char newChar = s.charAt(sequencePos);
				if (aChar.isValid(newChar)) {
					aChar.setValue(newChar);
					sequencePos = incrementSequencePos(sequencePos, maskPos, cursorPos);
					break;
				} else if (aChar.isOptional()) {
					Char correctChar = getMaskCharFor(newChar, maskPos + 1);
					if (correctChar != null) {
						maskPos = mMask.indexOf(correctChar);
						correctChar.setValue(newChar);
						sequencePos = incrementSequencePos(sequencePos, maskPos, cursorPos);
						break;
					}
				}
				sequencePos = incrementSequencePos(sequencePos, maskPos, cursorPos);
			}

			if (sequencePos >= s.length()) {
				break;
			}
		}
		maskPos++;
		for (; maskPos < mMask.size(); maskPos++) {
			mMask.get(maskPos).removeValue();
		}
	}

	private int incrementSequencePos(int sequencePos, int maskPos, int cursorPos) {
		sequencePos += 1;
		if (sequencePos == cursorPos) {
			mSelection = maskPos;
			while (mSelection + 1 < mMask.size() && mMask.get(mSelection + 1) instanceof Const) {
				mSelection++;
			}
			mSelection++;
		}
		return sequencePos;
	}

	private Char getMaskCharFor(char newChar, int maskPos) {
		if (maskPos < mMask.size()) {
			Char aChar = mMask.get(maskPos);
			if (aChar.isValid(newChar)) {
				return aChar;
			} else if (aChar.isOptional()) {
				return getMaskCharFor(newChar, maskPos + 1);
			}
		}
		return null;
	}

	CharSequence getText(boolean onlyValuable) {
		StringBuilder builder = new StringBuilder();
		boolean hasValue = false;
		for (Char c : mMask) {
			if (c.hasValue()) {
				if (!hasValue && !(c instanceof Const)) {
					hasValue = true;
				}
				if (!onlyValuable || c.isValuable()) {
					builder.append(c.getValue());
				}
			} else {
				break;
			}
		}
		if (hasValue || onlyValuable) {
			return builder;
		} else {
			return "";
		}
	}

	void changeSelection(int start) {
		mSelection = start;
	}

	int getSelection() {
		return mSelection;
	}

	public int getMaskLength() {
		return mMask.size();
	}
}
