package com.humandevice.maskwatcher;

import android.text.Editable;
import android.text.Selection;
import android.text.SpanWatcher;
import android.text.Spannable;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.widget.EditText;

/**
 * Klasa umożliwaiająca dodanie do {@link EditText} maski.
 * <p>
 * <b>Użycie:</b>
 * <pre><code>
   EditText editText = (EditText) findViewById(R.id.text1);
   InputMask mask = {@link InputMask#applyTo(EditText, String) InputMask.applyTo(editText, "(00)-(000)")};</code></pre>
 * <p>Jako drugi parametr metody {@link #applyTo(EditText, String)} należy podać maskę.</p>
 * <p>Znaki specjalne, które mogą zostać użyte w masce:<pre>
 * A - litera
 * a - litera (opcjonalna)
 * 0 - cyfra
 * 9 - cyfra (opcjonalna)
 * * - dowolny znak
 * ? - dowolny znak (opcjonalny)
 * () - grupa zawierająca znaczący tekst który może zostać pobrany poprzez wywołanie metody {@link #getText(boolean) #getText(true)}
 * \L - otwarcie grupy małych liter
 * \U - otwarcie grupy dużych liter
 * \E - zamknięcie grupy</pre>
 * <b>Przykłady:</b>
 * <pre>
 * - numer telefonu w formacie "777-77-77", maska: "000-00-00"
 * - skrót państwa dużymi literami "PL", maska: "\\UAA\\E"
 * - cena (2-3 cyfry całkowite) "50,00 zł", maska: "009,00 zł"
 *
 * <i>Aby pobrać tylko wybrane znaki należy je zawrzeć w nawiasach "()"
 * <b>Przykład:</b> dla maski: "(00)-(000)" i danych wejściowych "12-123"
 * metoda: {@link InputMask#getText(boolean) mask.getText(false)} zwróci "12-123"
 * metoda: {@link InputMask#getText(boolean) mask.getText(true)} zwróci "12123" </i>
 *
 * <b>Uwaga:</b> jeśli chce się wykorzystać znak specjalny jako stały to należy go poprzedzić znakiem "\\"
 * <b>Przykłady:</b>
 * dla "17a" gdzie "a" jest stałą należy wprowadzić, maska "00\\a"
 * dla "17\36", maska "00\\\\00"
 * dla "(029)777-77-77", maska: "\\(000\\)000-00-00"
 * </pre>
 *
 * <b>Uwaga:</b>
 * Niektóre klawiatury powodują złe działanie ograniczeń w przypadku włączonych sugestii, aby wyłączyć należy dodać:
 * <pre>   editText.setInputType(InputType.TYPE_TEXT_FLAG_NO_SUGGESTIONS);</pre>
 * <p>lub</p>
 * <pre>   android:inputType="textNoSuggestions"</pre>
 *
 * @author Rafal Zajfert
 */
public class InputMask implements TextWatcher {

	private final EditText mEditText;
	private MaskFormatter mMaskFormatter;

	private SpanWatcher mSpanWatcher = new SpanWatcher() {
		@Override
		public void onSpanAdded(final Spannable text, final Object what,
								final int start, final int end) {
		}

		@Override
		public void onSpanRemoved(final Spannable text, final Object what,
								  final int start, final int end) {
		}

		@Override
		public void onSpanChanged(final Spannable text, final Object what,
								  final int ostart, final int oend, final int nstart, final int nend) {
			if (what == Selection.SELECTION_START) {
				mMaskFormatter.changeSelection(nstart);
			}
		}
	};

	public InputMask(EditText editText, String mask) {
		mEditText = editText;

		mMaskFormatter = new MaskFormatter(mask);
	}

	public static InputMask applyTo(EditText editText, String mask) {
		InputMask watcher = new InputMask(editText, mask);
		editText.addTextChangedListener(watcher);
		return watcher;
	}


	@Override
	public void beforeTextChanged(CharSequence s, int start, int count, int after) {
		mEditText.getText().removeSpan(mSpanWatcher);
	}

	@Override
	public void onTextChanged(CharSequence s, int start, int before, int count) {
		if (mMaskFormatter.isEmptyMask()) {
			return;
		}
		mMaskFormatter.changeValue(s, start, before, count);
	}

	@Override
	public void afterTextChanged(Editable s) {
		if (mMaskFormatter.isEmptyMask()) {
			return;
		}
		CharSequence formattedString = mMaskFormatter.getText(false);

		if (!TextUtils.equals(s, formattedString)) {
			mEditText.removeTextChangedListener(this);
			s.replace(0, s.length(), formattedString);
			mEditText.addTextChangedListener(this);
		}

		mEditText.getText().setSpan(mSpanWatcher, 0, formattedString.length(), Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
		int selection = mMaskFormatter.getSelection();
		mEditText.setSelection(Math.max(0, Math.min(selection, formattedString.length())));
	}

	public void getText(boolean onlyValuable){
		mMaskFormatter.getText(onlyValuable);
	}
}
