package com.humandevice.maskwatcher;

import android.text.Editable;
import android.text.Selection;
import android.text.SpanWatcher;
import android.text.Spannable;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.widget.EditText;

/**
 * Klasa umożliwaiająca dodanie do {@link EditText} maski.
 * <p>
 * <b>Użycie:</b>
 * <pre><code>
 * EditText editText = (EditText) findViewById(R.id.text1);
 * InputMask mask = {@link InputMask#applyTo(EditText, String) InputMask.applyTo(editText, "(00)-(000)")};</code></pre>
 * <p>Jako drugi parametr metody {@link #applyTo(EditText, String)} należy podać maskę.</p>
 * <p>Znaki specjalne, które mogą zostać użyte w masce:<pre>
 * A - litera
 * a - litera (opcjonalna)
 * 0 - cyfra
 * 9 - cyfra (opcjonalna)
 * * - dowolny znak
 * ? - dowolny znak (opcjonalny)
 * () - grupa zawierająca znaczący tekst który może zostać pobrany poprzez wywołanie metody {@link #getText(boolean) #getText(true)}
 * \L - otwarcie grupy małych liter
 * \U - otwarcie grupy dużych liter
 * \E - zamknięcie grupy</pre>
 * <b>Przykłady:</b>
 * <pre>
 * - numer telefonu w formacie "777-77-77", maska: "000-00-00"
 * - skrót państwa dużymi literami "PL", maska: "\\UAA\\E"
 * - cena (2-3 cyfry całkowite) "50,00 zł", maska: "009,00 zł"
 *
 * <i>Aby pobrać tylko wybrane znaki należy je zawrzeć w nawiasach "()"
 * <b>Przykład:</b> dla maski: "(00)-(000)" i danych wejściowych "12-123"
 * metoda: {@link InputMask#getText(boolean) mask.getText(false)} zwróci "12-123"
 * metoda: {@link InputMask#getText(boolean) mask.getText(true)} zwróci "12123" </i>
 *
 * <b>Uwaga:</b> jeśli chce się wykorzystać znak specjalny jako stały to należy go poprzedzić znakiem "\\"
 * <b>Przykłady:</b>
 * dla "17a" gdzie "a" jest stałą należy wprowadzić, maska "00\\a"
 * dla "17\36", maska "00\\\\00"
 * dla "(029)777-77-77", maska: "\\(000\\)000-00-00"
 * </pre>
 * <p>
 * <b>Uwaga:</b>
 * Niektóre klawiatury powodują złe działanie ograniczeń w przypadku włączonych sugestii, aby wyłączyć należy dodać:
 * <pre>   editText.setInputType(InputType.TYPE_TEXT_FLAG_NO_SUGGESTIONS);</pre>
 * <p>lub</p>
 * <pre>   android:inputType="textNoSuggestions"</pre>
 *
 * <p><b>Uwaga 2:</b>
 * {@link EditText} musi mieć uprawnienie do dodawania wszystkich znaków z maski, w przeciwnym przypadku może zostać rzucony {@link Exception}
 * <p><b>Przykład:</b>
 * <p>Jeśli maska zawiera inne znaki niż cyfry np. "00 zł" lub "00 000" oraz zostanie dodany <code>android:inputType="number"</code>
 * to biblioteka rzuci błędem {@link IndexOutOfBoundsException} ponieważ znaki ' ', 'z' oraz 'ł' nie są cyframi.
 * Aby rozwiązać ten problem należy zmienić inputType lub zdefiniować dozwolone znaki poprzez <code>android:digits="1234567890łz "</code>
 *
 * @author Rafal Zajfert
 */
public class InputMask implements TextWatcher, SpanWatcher {
	
	private final EditText editText;
	private MaskFormatter maskFormatter;
	private boolean textChanging;
	private boolean removeAfterChange;
	
	/**
	 * Klasa umożliwaiająca dodanie do {@link EditText} maski.
	 * <p>
	 * <b>Użycie:</b>
	 * <pre><code>
	 * EditText editText = (EditText) findViewById(R.id.text1);
	 * InputMask mask = {@link InputMask#applyTo(EditText, String) InputMask.applyTo(editText, "(00)-(000)")};</code></pre>
	 * <p>Jako drugi parametr metody {@link #applyTo(EditText, String)} należy podać maskę.</p>
	 * <p>Znaki specjalne, które mogą zostać użyte w masce:<pre>
	 * A - litera
	 * a - litera (opcjonalna)
	 * 0 - cyfra
	 * 9 - cyfra (opcjonalna)
	 * * - dowolny znak
	 * ? - dowolny znak (opcjonalny)
	 * () - grupa zawierająca znaczący tekst który może zostać pobrany poprzez wywołanie metody {@link #getText(boolean) #getText(true)}
	 * \L - otwarcie grupy małych liter
	 * \U - otwarcie grupy dużych liter
	 * \E - zamknięcie grupy</pre>
	 * <b>Przykłady:</b>
	 * <pre>
	 * - numer telefonu w formacie "777-77-77", maska: "000-00-00"
	 * - skrót państwa dużymi literami "PL", maska: "\\UAA\\E"
	 * - cena (2-3 cyfry całkowite) "50,00 zł", maska: "009,00 zł"
	 *
	 * <i>Aby pobrać tylko wybrane znaki należy je zawrzeć w nawiasach "()"
	 * <b>Przykład:</b> dla maski: "(00)-(000)" i danych wejściowych "12-123"
	 * metoda: {@link InputMask#getText(boolean) mask.getText(false)} zwróci "12-123"
	 * metoda: {@link InputMask#getText(boolean) mask.getText(true)} zwróci "12123" </i>
	 *
	 * <b>Uwaga:</b> jeśli chce się wykorzystać znak specjalny jako stały to należy go poprzedzić znakiem "\\"
	 * <b>Przykłady:</b>
	 * dla "17a" gdzie "a" jest stałą należy wprowadzić, maska "00\\a"
	 * dla "17\36", maska "00\\\\00"
	 * dla "(029)777-77-77", maska: "\\(000\\)000-00-00"
	 * </pre>
	 * <p>
	 * <b>Uwaga:</b>
	 * Niektóre klawiatury powodują złe działanie ograniczeń w przypadku włączonych sugestii, aby wyłączyć należy dodać:
	 * <pre>   editText.setInputType(InputType.TYPE_TEXT_FLAG_NO_SUGGESTIONS);</pre>
	 * <p>lub</p>
	 * <pre>   android:inputType="textNoSuggestions"</pre>
	 *
	 * <p><b>Uwaga 2:</b>
	 * {@link EditText} musi mieć uprawnienie do dodawania wszystkich znaków z maski, w przeciwnym przypadku może zostać rzucony {@link Exception}
	 * <p><b>Przykład:</b>
	 * <p>Jeśli maska zawiera inne znaki niż cyfry np. "00 zł" lub "00 000" oraz zostanie dodany <code>android:inputType="number"</code>
	 * to biblioteka rzuci błędem {@link IndexOutOfBoundsException} ponieważ znaki ' ', 'z' oraz 'ł' nie są cyframi.
	 * Aby rozwiązać ten problem należy zmienić inputType lub zdefiniować dozwolone znaki poprzez <code>android:digits="1234567890łz "</code>
	 */
	private InputMask(EditText editText, String mask) {
		this.editText = editText;
		
		maskFormatter = new MaskFormatter(mask);
		if (editText.length() > 0)
			maskFormatter.changeValue(editText.getText(), 0, 0, editText.length());
	}
	
	public static InputMask applyTo(EditText editText, String mask) {
		InputMask watcher = new InputMask(editText, mask);
		editText.addTextChangedListener(watcher);
		return watcher;
	}
	
	public void removeMask() {
		if (!textChanging) {
			editText.removeTextChangedListener(this);
		} else {
			removeAfterChange = true;
		}
	}
	
	@Override
	public void beforeTextChanged(CharSequence s, int start, int count, int after) {
		textChanging = true;
		editText.getText().removeSpan(this);
	}
	
	@Override
	public void onTextChanged(CharSequence s, int start, int before, int count) {
		if (maskFormatter.isEmptyMask()) {
			return;
		}
		
		if (!removeAfterChange) {
			maskFormatter.changeValue(s, start, before, count);
		}
	}
	
	@Override
	public void afterTextChanged(Editable s) {
		if (maskFormatter.isEmptyMask()) {
			return;
		}
		CharSequence formattedString = maskFormatter.getText(false);
		
		if (!TextUtils.equals(s, formattedString)) {
			if (!removeAfterChange) {
				editText.removeTextChangedListener(this);
			}
			s.replace(0, s.length(), formattedString);
			if (!removeAfterChange) {
				editText.addTextChangedListener(this);
			}
		}
		
		Editable text = editText.getText();
		text.setSpan(this, 0, text.length(), Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
		if (!removeAfterChange) {
			editText.setSelection(Math.max(0, Math.min(maskFormatter.getSelection(), formattedString.length())));
		}
		textChanging = false;
		if (removeAfterChange) {
			editText.removeTextChangedListener(this);
			removeAfterChange = false;
		}
	}
	
	public CharSequence getText(boolean onlyValuable) {
		return maskFormatter.getText(onlyValuable);
	}
	
	@Override
	public void onSpanAdded(Spannable text, Object what, int start, int end) {
	}
	
	@Override
	public void onSpanRemoved(Spannable text, Object what, int start, int end) {
	}
	
	@Override
	public void onSpanChanged(Spannable text, Object what, int ostart, int oend, int nstart, int nend) {
		if (what == Selection.SELECTION_START && !removeAfterChange) {
			maskFormatter.changeSelection(nstart);
		}
	}
}
