package com.humandevice.android.loadingdialog;

/**
 * Created by danielc on 29.08.2016.
 */

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.support.annotation.StyleRes;
import android.view.View;
import android.widget.TextView;

import com.wang.avi.AVLoadingIndicatorView;

import java.util.Locale;

/**
 * Created by danielc on 25.08.2016.
 */
public class LoadingDialog extends Dialog {

    private AVLoadingIndicatorView mProgressView;
    private TextView mProgressTextView;
    private TextView mDescriptionTextView;
    private View mBackgroundView;

    private float mMaxProgress = 100;
    private float mProgress = 0;
    private ProgressFormatter mProgressFormatter = new ProgressFormatter() {

        private static final String PROGRESS_FORMAT = "%d%%";

        @Override
        public String format(float progress, float maxProgress) {
            int percent = (int) ((progress / maxProgress) * (float) 100 + 0.5);
            return String.format(Locale.getDefault(), PROGRESS_FORMAT, percent);
        }
    };

    public LoadingDialog(Activity context) {
        this(context, R.style.progressDialog);
    }

    public LoadingDialog(Context context, @StyleRes int theme) {
        this(context, theme, false, null);
    }

    public LoadingDialog(Context context, boolean cancelable, OnCancelListener cancelListener) {
        this(context, R.style.progressDialog, cancelable, cancelListener);
    }

    public LoadingDialog(Context context, @StyleRes int theme, boolean cancelable, OnCancelListener cancelListener) {
        super(context, theme);
        setCancelable(cancelable);
        setCanceledOnTouchOutside(cancelable);
        setOnCancelListener(cancelListener);

        init();

        if (theme > 0) {
            TypedArray typedArray = context.obtainStyledAttributes(theme, R.styleable.progressDialog);
            int color = typedArray.getColor(R.styleable.progressDialog_pd_backgroundColor, Color.RED);
            int descriptionColor = typedArray.getColor(R.styleable.progressDialog_pd_descriptionTextColor, Color.WHITE);
            int progressTextColor = typedArray.getColor(R.styleable.progressDialog_pd_progressTextColor, Color.WHITE);
            int progressColor = typedArray.getColor(R.styleable.progressDialog_pd_progressColor, Color.WHITE);
            boolean indeterminate = typedArray.getBoolean(R.styleable.progressDialog_pd_indeterminate, true);

            float alpha = typedArray.getFloat(R.styleable.progressDialog_pd_backgroundAlpha, 1f);
            setBackgroundColor(color);
            setDescriptionTextColor(descriptionColor);
            setProgressTextColor(progressTextColor);
            setProgressColor(progressColor);
            setBackgroundAlpha(alpha);
            setIndeterminate(indeterminate);

            typedArray.recycle();
        }
    }

    private void init() {
        setContentView();
        mProgressView = (AVLoadingIndicatorView) findViewById(R.id.progress);
        mProgressTextView = (TextView) findViewById(R.id.progress_text);
        mDescriptionTextView = (TextView) findViewById(R.id.progress_description);
        mBackgroundView = findViewById(R.id.background);

        mProgressView.setBackgroundColor(Color.TRANSPARENT);
        setMaxProgress(100);
        setProgress(0);
        setIndeterminate(true);
        setDescriptionText("");
    }

    public void setIndeterminate(boolean indeterminate) {
        mProgressTextView.setVisibility(indeterminate ? View.GONE : View.VISIBLE);
    }

    public void setBackgroundAlpha(float alpha) {
        mBackgroundView.setAlpha(alpha);
    }

    public void setBackgroundColor(int color) {
        mBackgroundView.setBackgroundColor(color);
    }

    public void setProgressColor(int color) {
        mProgressView.setIndicatorColor(color);
    }

    public float getProgress() {
        return mProgress;
    }

    public void setProgress(float progress) {
        mProgress = progress;
        setProgressText();
    }

    public float getMaxProgress() {
        return mMaxProgress;
    }

    public void setMaxProgress(float maxProgress) {
        mMaxProgress = maxProgress;
    }

    public void setDescriptionTextColor(int color) {
        mDescriptionTextView.setTextColor(color);
    }

    public void setProgressTextColor(int color) {
        mProgressTextView.setTextColor(color);
    }

    public void setDescriptionText(String text) {
        mDescriptionTextView.setText(text);
    }

    public void setDescriptionText(@StringRes int textRes) {
        mDescriptionTextView.setText(textRes);
    }

    private void setProgressText() {
        mProgressTextView.setText(mProgressFormatter.format(mProgress, getMaxProgress()));
    }

    public void setProgressFormatter(@NonNull ProgressFormatter formatter) {
        this.mProgressFormatter = formatter;
    }

    public void setContentView() {
        super.setContentView(R.layout.dialog_loading);
    }


    public interface ProgressFormatter {
        String format(float progress, float maxProgress);
    }


}
