package com.humandevice.android.locationtool;

import android.content.Context;
import android.location.Location;
import android.location.LocationManager;
import android.os.Build;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresPermission;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.location.LocationSettingsRequest;
import com.google.android.gms.location.LocationSettingsResult;
import com.google.android.gms.location.LocationSettingsStatusCodes;
import com.rafalzajfert.androidlogger.Logger;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Service that provide location information from GPS or Network
 *
 * @author Rafal Orlik
 * @date 2015-12-16
 */

public class LocationService {

	private static LocationService sInstance;
	private static LocationServiceConfiguration mConfiguration = null;

	public static void init(@NonNull LocationServiceConfiguration configuration) {
		mConfiguration = configuration;
	}

	/**
	 * returns sInstance of Location tracker
	 */
	@NonNull
	public static LocationService getInstance() {
		if (mConfiguration == null) {
			throw new RuntimeException("LocationService was not initialized!");
		}

		if (sInstance == null) {
			sInstance = new LocationService(mConfiguration.mContext);
		}
		return sInstance;
	}

	private LocationCoreService mCoreService;

	public LocationService(Context context) {
		mCoreService = new LocationCoreService(context);
	}

	@RequiresPermission(anyOf = {"android.permission.ACCESS_COARSE_LOCATION", "android.permission.ACCESS_FINE_LOCATION"})
	public void start() {
		mCoreService.start();
	}

	public void stop() {
		mCoreService.stop();
	}

	public void addLocationChangeListener(@NonNull LocationChangeListener listener) {
		mCoreService.addLocationChangeListener(listener);
	}

	public void removeLocationChangeListener(@NonNull LocationChangeListener listener) {
		mCoreService.removeLocationChangeListener(listener);
	}

	public boolean isGpsEnabled() {
		return mCoreService.isGpsEnabled();
	}

	public boolean isLocationEnabled() {
		return mCoreService.isLocationEnabled();
	}

	public void checkLocationState(@NonNull LocationStateListener locationStateListener) {
		mCoreService.connect();
		mCoreService.checkLocationSettings(locationStateListener);
	}

	@RequiresPermission(anyOf = {"android.permission.ACCESS_COARSE_LOCATION", "android.permission.ACCESS_FINE_LOCATION"})
	public void getCurrentLocation(LocationChangeListener oneTimeListener) {
		mCoreService.connect();
		@SuppressWarnings("MissingPermission")
		Location currentLocation = mCoreService.getCurrentLocation();
		if (currentLocation != null) {
			oneTimeListener.onLocationChange(currentLocation);
			mCoreService.stopIfNotConnected();
		} else {
			mCoreService.addLocationChangeListener(new SingleTimeListener(oneTimeListener));
		}
//		mCoreService.addLocationChangeListener(new SingleTimeListener(oneTimeListener));
	}

	private class SingleTimeListener implements LocationChangeListener {

		private LocationChangeListener mListener;

		public SingleTimeListener(LocationChangeListener listener) {
			mListener = listener;
		}

		@Override
		public void onLocationChange(@NonNull Location location) {
			mListener.onLocationChange(location);
			mCoreService.stopIfNotConnected();
		}
	}

	private class LocationCoreService implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener, LocationListener {

		private final Context mContext;
		private final GoogleApiClient mGoogleApiClient;
		private final Set<LocationChangeListener> mListeners = new HashSet<>();

		private boolean mIsApiConnected;
		private LocationRequest mLocationRequest;

		private LocationCoreService(@NonNull Context context) {
			mContext = context;
			mGoogleApiClient = new GoogleApiClient.Builder(context)
					.addApi(LocationServices.API)
					.addConnectionCallbacks(this)
					.addOnConnectionFailedListener(this)
					.build();
			mLocationRequest = LocationRequest.create()
					.setPriority(mConfiguration.getPriority())
					.setInterval(mConfiguration.getUpdateInterval())
					.setFastestInterval(mConfiguration.getFastestInterval());
		}

		public boolean isGpsEnabled() {
			boolean enabled = false;
			try {
				if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
					int locationMode = Settings.Secure.getInt(mContext.getContentResolver(), Settings.Secure.LOCATION_MODE);
					enabled = locationMode == Settings.Secure.LOCATION_MODE_HIGH_ACCURACY;
				} else {
					LocationManager locationManager = (LocationManager) mContext.getSystemService(Context.LOCATION_SERVICE);
					enabled = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER);
				}
			} catch (Settings.SettingNotFoundException e) {
				Logger.error(e);
			}
			return enabled;
		}

		public boolean isLocationEnabled() {
			boolean enabled = false;
			try {
				if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
					int locationMode = Settings.Secure.getInt(mContext.getContentResolver(), Settings.Secure.LOCATION_MODE);
					enabled = locationMode != Settings.Secure.LOCATION_MODE_OFF;
				} else {
					LocationManager locationManager = (LocationManager) mContext.getSystemService(Context.LOCATION_SERVICE);
					enabled = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER) || locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER);
				}
			} catch (Settings.SettingNotFoundException e) {
				Logger.error(e);
			}
			return enabled;
		}

		@Override
		public void onConnected(Bundle bundle) {
			startLocationUpdates();
		}

		/**
		 * Create the location request and registers for updates.
		 */
		private void startLocationUpdates() {
			LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder()
					.addLocationRequest(mLocationRequest)
					.setAlwaysShow(true);

			PendingResult<LocationSettingsResult> result = LocationServices.SettingsApi.checkLocationSettings(mGoogleApiClient, builder.build());
			result.setResultCallback(new ResultCallback<LocationSettingsResult>() {
				@Override
				public void onResult(@NonNull LocationSettingsResult result) {
					final Status status = result.getStatus();
					switch (status.getStatusCode()) {
						case LocationSettingsStatusCodes.SUCCESS:
							Logger.debug("onResult(SUCCESS)");
							//noinspection MissingPermission
							LocationServices.FusedLocationApi.requestLocationUpdates(mGoogleApiClient, mLocationRequest, LocationCoreService.this);
							break;
						case LocationSettingsStatusCodes.RESOLUTION_REQUIRED:
							Logger.debug("onResult(RESOLUTION_REQUIRED)");
							break;
						case LocationSettingsStatusCodes.SETTINGS_CHANGE_UNAVAILABLE:
							Logger.debug("onResult(SETTINGS_CHANGE_UNAVAILABLE)");
							break;
					}
				}
			});
		}

		private void checkLocationSettings(@NonNull final LocationStateListener listener) {
			if (mLocationRequest != null) {
				LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder()
						.addLocationRequest(mLocationRequest)
						.setAlwaysShow(true);

				PendingResult<LocationSettingsResult> result = LocationServices.SettingsApi.checkLocationSettings(mGoogleApiClient, builder.build());
				result.setResultCallback(new ResultCallback<LocationSettingsResult>() {
					@Override
					public void onResult(@NonNull LocationSettingsResult result) {
						final Status status = result.getStatus();
						switch (status.getStatusCode()) {
							case LocationSettingsStatusCodes.SUCCESS:
								Logger.debug("onResult(SUCCESS)");
								listener.onLocationEnabled();
								stopIfNotConnected();

								break;
							case LocationSettingsStatusCodes.RESOLUTION_REQUIRED:
								Logger.debug("onResult(RESOLUTION_REQUIRED)");
								listener.onLocationDisabled(status);
								stopIfNotConnected();

								break;
							case LocationSettingsStatusCodes.SETTINGS_CHANGE_UNAVAILABLE:
								Logger.error("onResult(SETTINGS_CHANGE_UNAVAILABLE)");
								break;
						}
					}
				});
			} else {
				Logger.warning("No location request yet");
			}
		}


		@Override
		public void onConnectionSuspended(int i) {
			Logger.warning("onConnectionSuspended", i);
		}

		@Override
		public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
			Logger.warning("onConnectionFailed", connectionResult.getErrorMessage());
		}

		@Override
		public void onLocationChanged(Location location) {
			if (location != null) {
				notifyListeners(location);
			}
		}

		@RequiresPermission(anyOf = {"android.permission.ACCESS_COARSE_LOCATION", "android.permission.ACCESS_FINE_LOCATION"})
		@Nullable
		public Location getCurrentLocation() {
			//noinspection MissingPermission - przeciez wyzej sprawdzam
			return LocationServices.FusedLocationApi.getLastLocation(mGoogleApiClient);
		}

		/**
		 * Send location to all mListeners
		 */
		synchronized private void notifyListeners(@NonNull Location location) {
			Iterator<LocationChangeListener> iterator = mListeners.iterator();
			while (iterator.hasNext()) {
				LocationChangeListener listener = iterator.next();
				if (listener instanceof SingleTimeListener) {
					iterator.remove();
				}
				listener.onLocationChange(location);
			}
		}

		/**
		 * Starts LocationService to receive location updates.
		 */
		public void start() {
			connect();
			mIsApiConnected = true;
		}

		private void connect() {
			if (!mGoogleApiClient.isConnected()) {
				mGoogleApiClient.connect();
			}
		}

		/**
		 * Stops location updates subscription
		 */
		public boolean stop() {
			mIsApiConnected = false;
			boolean disconnect = disconnect();
			//after updates stopped remove single listeners
			removeSingleListeners();
			return disconnect;
		}

		public void stopIfNotConnected() {
			if (!mIsApiConnected && mListeners.size() == 0) {
				disconnect();
			}
		}

		private boolean disconnect() {
			if (mGoogleApiClient.isConnected()) {
				LocationServices.FusedLocationApi.removeLocationUpdates(mGoogleApiClient, this);
				mGoogleApiClient.disconnect();
				return true;
			}
			return false;
		}

		synchronized private void removeSingleListeners() {
			Iterator<LocationChangeListener> iterator = mListeners.iterator();
			while (iterator.hasNext()) {
				LocationChangeListener listener = iterator.next();
				if (listener instanceof SingleTimeListener) {
					iterator.remove();
				}
			}
		}

		/**
		 * Add listener for receiving notifications when the location has changed.
		 * <br/><br/>
		 * <b>Remember</b> to {@link #removeLocationChangeListener(LocationChangeListener) remove} this listener when is not longer needed
		 */
		synchronized public void addLocationChangeListener(@NonNull LocationChangeListener listener) {
			mListeners.add(listener);
		}

		/**
		 * Remove specified listener used for receiving notifications
		 */
		synchronized public void removeLocationChangeListener(@NonNull LocationChangeListener listener) {
			mListeners.remove(listener);
		}
	}

	/**
	 * Used for receiving notifications from the LocationService when the location has changed.
	 */
	public interface LocationChangeListener {
		void onLocationChange(@NonNull Location location);
	}

	public interface LocationStateListener {
		void onLocationDisabled(Status status);

		void onLocationEnabled();
	}
}
