package com.humandevice.android.locationtool;

import android.Manifest;
import android.content.Context;
import android.location.Location;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresPermission;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.location.LocationSettingsRequest;
import com.google.android.gms.location.LocationSettingsResult;
import com.google.android.gms.location.LocationSettingsStatusCodes;
import com.rafalzajfert.androidlogger.Logger;

import java.util.Collections;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

/**
 * TODO Dokumentacja
 *
 * @author Rafał Orlik
 * @date 2016-10-04
 */

class CoreLocationTool implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener {

	private final GoogleApiClient mGoogleClient;
	private final Map<LocationListener, LocationRequest> mLocationRequestMap;

	CoreLocationTool(@NonNull Context context) {
		mGoogleClient = new GoogleApiClient.Builder(context)
				.addOnConnectionFailedListener(this)
				.addConnectionCallbacks(this)
				.addApi(LocationServices.API)
				.build();
		mLocationRequestMap = Collections.synchronizedMap(new Hashtable<LocationListener, LocationRequest>());
	}

	@Override
	public void onConnected(@Nullable Bundle bundle) {
		//noinspection MissingPermission
		startLocationUpdates();
	}

	@Override
	public void onConnectionSuspended(int i) {
		Logger.warning("onConnectionSuspended", i);
	}

	@Override
	public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
		Logger.warning("onConnectionFailed", connectionResult.getErrorMessage());
	}

	/**
	 * Starts LocationService to receive location updates.
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	void start() {
		connect();
	}

	/**
	 * Stops location updates subscription
	 */
	boolean stop() {
		stopLocationUpdates();
		return disconnect();
	}

	private void connect() {
		mGoogleClient.connect();
	}

	/**
	 * @return true if service was disconnected and false if there was no need for that
	 */
	private boolean disconnect() {
		if (mGoogleClient.isConnected()) {
			mGoogleClient.disconnect();
			return true;
		}
		return false;
	}

	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	private void startLocationUpdates() {
		Set<LocationListener> locationListeners = mLocationRequestMap.keySet();
		for (LocationListener listener : locationListeners) {
			//noinspection MissingPermission
			LocationServices.FusedLocationApi.requestLocationUpdates(mGoogleClient, mLocationRequestMap.get(listener), listener);
		}
	}

	private void stopLocationUpdates() {
		Set<LocationListener> locationListeners = mLocationRequestMap.keySet();
		for (LocationListener listener : locationListeners) {
			LocationServices.FusedLocationApi.removeLocationUpdates(mGoogleClient, listener);
		}
	}

	/**
	 * <p>If a location is not available, which should happen very rarely, null will be returned. The best accuracy available while respecting the location permissions will be returned.</p>
	 * @return Returns the best most recent location currently available.
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	Location getLastLocation() {
		//noinspection MissingPermission
		return LocationServices.FusedLocationApi.getLastLocation(mGoogleClient);
	}

	/**
	 * <p>Adds location request. If you add new request with this same {@link LocationListener} the old request will be replaced.</p>
	 * <p>Please have in mind that this method will start location updates if location service meets request settings</p>
	 * <p>Remember to cancel your request with {@link LocationTool#cancelLocationUpdates} when you no longer need location updates</p>
	 * @param locationRequest  I suggest using {@link LocationRequestBuilder} to create location requests
	 * @param locationListener You can pass here {@link ToolLocationListener} that allows you ask to turn on gsp
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	void requestLocationUpdates(final LocationRequest locationRequest, final LocationListener locationListener) {
		mLocationRequestMap.put(locationListener, locationRequest);
		LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder().addLocationRequest(locationRequest).setAlwaysShow(true);

		PendingResult<LocationSettingsResult> result = LocationServices.SettingsApi.checkLocationSettings(mGoogleClient, builder.build());
		result.setResultCallback(new ResultCallback<LocationSettingsResult>() {
			@Override
			public void onResult(@NonNull LocationSettingsResult result) {
				final Status status = result.getStatus();
				switch (status.getStatusCode()) {
					case LocationSettingsStatusCodes.SUCCESS:
						if (mGoogleClient.isConnected()) {
							//noinspection MissingPermission
							startLocationUpdates();
						} else {
							//noinspection MissingPermission
							start();
						}
						// All location settings are satisfied. The client can
						// initialize location requests here.
						break;
					case LocationSettingsStatusCodes.RESOLUTION_REQUIRED:
						if (locationListener instanceof ToolLocationListener) {
							((ToolLocationListener) locationListener).onResolutionRequired(status);
						}
						break;
					case LocationSettingsStatusCodes.SETTINGS_CHANGE_UNAVAILABLE:
						Logger.error("SETTINGS_CHANGE_UNAVAILABLE");
						// Location settings are not satisfied. However, we have no way
						// to fix the settings so we won't show the dialog.
						break;
				}
			}
		});
	}

	/**
	 * <p>Used to unregister listener from any further location updates.</p>
	 * @param locationListener listener to unregister
	 * @return true if there was listener to remove
	 */
	boolean cancelLocationUpdates(LocationListener locationListener) {
		LocationServices.FusedLocationApi.removeLocationUpdates(mGoogleClient, locationListener);
		return mLocationRequestMap.remove(locationListener) != null;
	}

	void cancelAllLocationUpdates(){
		stopLocationUpdates();
		mLocationRequestMap.clear();
	}

}
