package com.humandevice.android.locationtool;

import android.Manifest;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresPermission;
import android.support.v4.app.NotificationCompat;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.location.LocationSettingsRequest;
import com.google.android.gms.location.LocationSettingsResult;
import com.google.android.gms.location.LocationSettingsStatusCodes;

import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import software.rsquared.androidlogger.Logger;

import static android.content.Context.NOTIFICATION_SERVICE;

/**
 * TODO Dokumentacja
 *
 * @author Rafał Orlik
 * @date 2016-10-04
 */

class CoreLocationTool implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener {

	private final GoogleApiClient googleClient;
	private final Map<LocationListener, LocationRequest> locationRequestMap;
	private final Map<LocationListener, LocationRequest> continuousLocationRequestMap;
	@Nullable
	private ConnectionListener connectionListener;

	CoreLocationTool(@NonNull Context context) {
		googleClient = new GoogleApiClient.Builder(context)
				.addOnConnectionFailedListener(this)
				.addConnectionCallbacks(this)
				.addApi(LocationServices.API)
				.build();
		locationRequestMap = Collections.synchronizedMap(new Hashtable<LocationListener, LocationRequest>());
		continuousLocationRequestMap = Collections.synchronizedMap(new Hashtable<LocationListener, LocationRequest>());
	}

	@Override
	public void onConnected(@Nullable Bundle bundle) {
		//noinspection MissingPermission
		startLocationUpdates();
		if (connectionListener != null) {
			connectionListener.onConnectionSuccess();
		}
		connectionListener = null;
	}

	@Override
	public void onConnectionSuspended(int i) {
		Logger.warning("onConnectionSuspended", i);
	}

	@Override
	public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
		Logger.warning("onConnectionFailed", connectionResult.getErrorMessage(), connectionResult);
		try {
			if (connectionResult.getErrorCode() == ConnectionResult.SERVICE_VERSION_UPDATE_REQUIRED
					|| connectionResult.getErrorCode() == ConnectionResult.SERVICE_MISSING) {
				Context context = googleClient.getContext();
				Intent intent = new Intent(Intent.ACTION_VIEW).setData(Uri.parse("https://play.google.com/store/apps/details?id=com.google.android.gms"));

				NotificationCompat.Builder builder = new NotificationCompat.Builder(context)
						.setPriority(Notification.PRIORITY_HIGH)
						.setAutoCancel(true)
						.setDefaults(Notification.DEFAULT_ALL)
						.setSmallIcon(R.drawable.googleg_disabled_color_18)
						.setContentTitle(context.getString(R.string.notification_services_error_title))
						.setStyle(new NotificationCompat.BigTextStyle().bigText(context.getString(R.string.notification_services_error_message)))
						.setContentText(context.getString(R.string.notification_services_error_message))
						.setContentIntent(PendingIntent.getActivity(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT));

				((NotificationManager) context.getSystemService(NOTIFICATION_SERVICE)).notify("location", 44, builder.build());
			}
		} catch (Exception e) {
			Logger.error(e);
		}
		if (connectionListener != null) {
			connectionListener.onConnectionFailed();
		}
		connectionListener = null;
	}

	/**
	 * Starts LocationService to receive location updates.
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	void start() {
		connect();
	}

	/**
	 * Starts LocationService to receive location updates.
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	void start(ConnectionListener listener) {
		connectionListener = listener;
		connect();
	}

	/**
	 * Stops only {@link UpdateType#WEAK} location updates subscription
	 *
	 * @return true if no {@link UpdateType#CONTINUOUS}
	 */
	boolean stop() {
		stopWeakLocationUpdates();
		Logger.info("stop location updates");
		boolean disconnect = false;
		if (continuousLocationRequestMap.isEmpty()) {
			disconnect = disconnect();
		}
		Logger.info("remaining continuous updates", continuousLocationRequestMap.size());
		return disconnect;
	}

	private void connect() {
		Logger.debug(googleClient.isConnected(), googleClient.isConnecting());
		googleClient.connect();
		Logger.debug("connecting");
	}

	/**
	 * @return true if service was disconnected and false if there was no need for that
	 */
	private boolean disconnect() {
		if (googleClient.isConnected()) {
			googleClient.disconnect();
			Logger.debug("disconnected");
			return true;
		}
		Logger.debug("no need to disconnect");
		return false;
	}

	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	private void startLocationUpdates() {
		if (googleClient.isConnected()) {
			for (LocationListener listener : locationRequestMap.keySet()) {
				//noinspection MissingPermission
				LocationServices.FusedLocationApi.requestLocationUpdates(googleClient, locationRequestMap.get(listener), listener);
			}
			for (LocationListener listener : continuousLocationRequestMap.keySet()) {
				//noinspection MissingPermission
				LocationServices.FusedLocationApi.requestLocationUpdates(googleClient, continuousLocationRequestMap.get(listener), listener);
			}
		}
	}

	private void stopWeakLocationUpdates() {
		if (googleClient.isConnected()) {
			for (LocationListener listener : locationRequestMap.keySet()) {
				LocationServices.FusedLocationApi.removeLocationUpdates(googleClient, listener);
			}
		}
	}

	/**
	 * <p>If a location is not available, which should happen very rarely, null will be returned. The best accuracy available while respecting the location permissions will be returned.</p>
	 *
	 * @return Returns the best most recent location currently available.
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	Location getLastLocation() {
		//noinspection MissingPermission
		return LocationServices.FusedLocationApi.getLastLocation(googleClient);
	}

	/**
	 * <p>Adds location request. If you add new request with this same {@link LocationListener} the old request will be replaced.</p>
	 * <p>Please have in mind that this method will start location updates if location service meets request settings</p>
	 * <p>Remember to cancel your request with {@link LocationTool#cancelLocationUpdates} when you no longer need location updates</p>
	 *
	 * @param locationRequest  I suggest using {@link LocationRequestBuilder} to create location requests
	 * @param locationListener You can pass here {@link ToolLocationListener} that allows you ask to turn on gsp
	 */
	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION})
	void requestLocationUpdates(final LocationRequest locationRequest, final LocationListener locationListener, @NonNull UpdateType updateType) {
		addLocationRequest(locationRequest, locationListener, updateType);
		LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder().addLocationRequest(locationRequest).setAlwaysShow(true);

		PendingResult<LocationSettingsResult> result = LocationServices.SettingsApi.checkLocationSettings(googleClient, builder.build());
		result.setResultCallback(new ResultCallback<LocationSettingsResult>() {
			@Override
			public void onResult(@NonNull LocationSettingsResult result) {
				final Status status = result.getStatus();
				Logger.verbose("checkLocationSettings", status.getStatusCode(), status.getStatusMessage());
				switch (status.getStatusCode()) {
					case LocationSettingsStatusCodes.SUCCESS:
						if (googleClient.isConnected()) {
							//noinspection MissingPermission
							startLocationUpdates();
						} else {
							//noinspection MissingPermission
							start();
						}
						// All location settings are satisfied. The client can
						// initialize location requests here.
						break;
					case LocationSettingsStatusCodes.RESOLUTION_REQUIRED:
						if (locationListener instanceof ToolLocationListener) {
							((ToolLocationListener) locationListener).onResolutionRequired(status);
						}
						break;
					case LocationSettingsStatusCodes.SETTINGS_CHANGE_UNAVAILABLE:
						Logger.error("SETTINGS_CHANGE_UNAVAILABLE");
						// Location settings are not satisfied. However, we have no way
						// to fix the settings so we won't show the dialog.
						break;
				}
			}
		});
	}

	/**
	 * <p>Used to unregister connectionListener from any further location updates.</p>
	 *
	 * @param locationListener connectionListener to unregister
	 * @return true if there was connectionListener to remove
	 */
	boolean cancelLocationUpdates(LocationListener locationListener) {
		if (googleClient.isConnected()) {
			LocationServices.FusedLocationApi.removeLocationUpdates(googleClient, locationListener);
		}
		return removeLocationRequest(locationListener);
	}

	/**
	 * <p>Unregisters all {@link UpdateType#WEAK} location updates</p>
	 */
	void cancelAllWeakLocationUpdates() {
		stopWeakLocationUpdates();
		locationRequestMap.clear();
	}

	@RequiresPermission(anyOf = {Manifest.permission.ACCESS_FINE_LOCATION})
	void addGeofence(GeofencingRequest request, PendingIntent pendingIntent) {
		//noinspection MissingPermission
		PendingResult<Status> statusPendingResult = LocationServices.GeofencingApi.addGeofences(googleClient, request, pendingIntent);
		statusPendingResult.setResultCallback(new ResultCallback<Status>() {
			@Override
			public void onResult(@NonNull Status status) {
				Logger.verbose(GeofenceStatusCodes.getStatusCodeString(status.getStatusCode()));

			}
		});
	}

	void removeGeofences(List<String> fenceIds) {
		PendingResult<Status> statusPendingResult = LocationServices.GeofencingApi.removeGeofences(googleClient, fenceIds);
		statusPendingResult.setResultCallback(new ResultCallback<Status>() {
			@Override
			public void onResult(@NonNull Status status) {
				Logger.verbose(GeofenceStatusCodes.getStatusCodeString(status.getStatusCode()));
			}
		});
	}

	void removeGeofences(PendingIntent pendingIntent) {
		PendingResult<Status> statusPendingResult = LocationServices.GeofencingApi.removeGeofences(googleClient, pendingIntent);
		statusPendingResult.setResultCallback(new ResultCallback<Status>() {
			@Override
			public void onResult(@NonNull Status status) {
				Logger.verbose(GeofenceStatusCodes.getStatusCodeString(status.getStatusCode()));
			}
		});
	}

	/**
	 * Add or replace locationRequest for given locationListener. If locationListener was already in use but with different {@link UpdateType} old usage will be removed.
	 *
	 * @param locationRequest
	 * @param locationListener
	 * @param updateType
	 */
	private void addLocationRequest(LocationRequest locationRequest, @NonNull LocationListener locationListener, @NonNull UpdateType updateType) {
		switch (updateType) {
			case CONTINUOUS:
				if (locationRequestMap.containsKey(locationListener)) {
					locationRequestMap.remove(locationListener);
				}
				continuousLocationRequestMap.put(locationListener, locationRequest);
				break;
			case WEAK:
			default:
				if (continuousLocationRequestMap.containsKey(locationListener)) {
					continuousLocationRequestMap.remove(locationListener);
				}
				locationRequestMap.put(locationListener, locationRequest);
				break;
		}
	}

	private boolean removeLocationRequest(@NonNull LocationListener locationListener) {
		boolean removed = locationRequestMap.remove(locationListener) != null;
		boolean removedCon = continuousLocationRequestMap.remove(locationListener) != null;
		return removed || removedCon;
	}

}
