package com.humandevice.android.mvpframework;

import android.os.Bundle;

import com.humandevice.android.core.layers.LayerView;


/**
 * @date 18.02.2016
 */
public class PresenterDelegate<V extends LayerView, P extends Presenter<V>> {
    private static final String PRESENTER_INDEX_KEY = "presenter-index";

    private P mPresenter;
    private long mPresenterId;
    private boolean mDestroyedBySystem;

    public void onCreate(PresenterCache cache, Bundle savedInstanceState, Class<? extends P> presenter) {
        if (savedInstanceState == null) {
            mPresenterId = cache.generateId();
        } else {
            mPresenterId = savedInstanceState.getLong(PRESENTER_INDEX_KEY);
        }
        mPresenter = cache.getPresenter(mPresenterId);
        if (mPresenter == null) {
            mPresenter = PresenterFactory.create(presenter);
            cache.setPresenter(mPresenterId, mPresenter);
        }
        PresenterBundle bundle = PresenterBundleUtils.getPresenterBundle(savedInstanceState, mPresenterId);
        mPresenter.onCreate(bundle);
    }

    public void onViewCreated(V view) {
        mPresenter.onBindView(view);
    }

    public void onDestroyView() {
        mPresenter.onUnbindView();
    }

    public void onResume() {
        mDestroyedBySystem = false;
        mPresenter.onResume();
    }

    public void onPause() {
        mPresenter.onPause();
    }

    public void onSaveInstanceState(Bundle outState) {
        mDestroyedBySystem = true;
        outState.putLong(PRESENTER_INDEX_KEY, mPresenterId);
        PresenterBundle bundle = new PresenterBundle();
        mPresenter.onSaveInstanceState(bundle);
        PresenterBundleUtils.setPresenterBundle(outState, bundle, mPresenterId);
    }

    public void onDestroy(PresenterCache cache) {
        if (!mDestroyedBySystem) {
            // User is exiting this view, remove presenter from the cache
            cache.setPresenter(mPresenterId, null);
            mPresenter.onDestroy();
        }
    }

    public P getPresenter() {
        return mPresenter;
    }

}
