package com.humandevice.android.mvpframework;

import android.content.Context;
import android.os.Bundle;
import android.support.annotation.CallSuper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;

import com.humandevice.android.core.layers.LayerView;
import com.humandevice.android.core.view.ToolsFragment;

/**
 * @date 18.02.2016
 */
public abstract class PresenterFragment<V extends LayerView, P extends Presenter<V>> extends ToolsFragment implements LayerView {
    private PresenterCache mPresenterCache;
    private PresenterDelegate<V, P> mPresenterDelegate = new PresenterDelegate<>();

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mPresenterDelegate.onCreate(mPresenterCache, savedInstanceState, getPresenterClass());
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        initView(view);
    }

    @Override
    public void onViewStateRestored(Bundle savedInstanceState) {
        super.onViewStateRestored(savedInstanceState);
        //noinspection unchecked
        mPresenterDelegate.onViewCreated((V) this);
    }

    @CallSuper
    @Override
    public void onAttachToFragment(Context context) {
        super.onAttachToFragment(context);
        checkActivityClass(context);
    }

    @Override
    public void onResume() {
        super.onResume();
        mPresenterDelegate.onResume();
    }

    @Override
    public void onPause() {
        super.onPause();
        mPresenterDelegate.onPause();
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        mPresenterDelegate.onSaveInstanceState(outState);
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        mPresenterDelegate.onDestroyView();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mPresenterDelegate.onDestroy(mPresenterCache);
    }


    private void checkActivityClass(Context context) {
        if (context instanceof PresenterCache) {
            mPresenterCache = (PresenterCache) context;
        } else {
            throw new RuntimeException(getClass().getSimpleName() + " must be attached to an Activity that implements "
                    + PresenterCache.class.getSimpleName());
        }
    }

    public P getPresenter() {
        return mPresenterDelegate.getPresenter();
    }

    /**
     * Get Class of the presenter implementation
     */
    public abstract Class<? extends P> getPresenterClass();

    /**
     * Find all views in root view
     */
    protected abstract void initView(View view);

}
