package com.humandevice.android.v4.mvpframework;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.DialogFragment;
import android.view.View;

import com.humandevice.android.core.layers.LayerView;
import com.humandevice.android.mvpframework.Presenter;
import com.humandevice.android.mvpframework.PresenterCache;
import com.humandevice.android.mvpframework.PresenterDelegate;

/**
 * @date 18.02.2016
 */
public abstract class PresenterDialogFragment<V extends LayerView, P extends Presenter<V>> extends DialogFragment {
    private PresenterCache mPresenterCache;
    private PresenterDelegate<V, P> mPresenterDelegate = new PresenterDelegate<>();

    @SuppressWarnings("deprecation")
    @Override
    public final void onAttach(Activity activity) {
        super.onAttach(activity);
        checkActivityClass(activity);
        onAttachToFragment(activity);
    }

    @Override
    public final void onAttach(Context context) {
        super.onAttach(context);
        checkActivityClass(context);
        onAttachToFragment(context);
    }

    /**
     * @param context context of activity that attached fragment
     */
    public void onAttachToFragment(Context context) {
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mPresenterDelegate.onCreate(mPresenterCache, savedInstanceState, getPresenterClass());
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        initView(view);
    }

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        Dialog dialog = createDialog(savedInstanceState);
        //noinspection unchecked
        mPresenterDelegate.onViewCreated((V) this);
        return dialog;
    }

    @Override
    public void onResume() {
        super.onResume();
        mPresenterDelegate.onResume();
    }

    @Override
    public void onPause() {
        super.onPause();
        mPresenterDelegate.onPause();
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        mPresenterDelegate.onSaveInstanceState(outState);
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        mPresenterDelegate.onDestroyView();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mPresenterDelegate.onDestroy(mPresenterCache);
    }

    private void checkActivityClass(Context context) {
        if (context instanceof PresenterCache) {
            mPresenterCache = (PresenterCache) context;
        } else {
            throw new RuntimeException(getClass().getSimpleName() + " must be attached to an Activity that implements "
                    + PresenterCache.class.getSimpleName());
        }
    }


    public P getPresenter() {
        return mPresenterDelegate.getPresenter();
    }

    /**
     * Get Class of the presenter implementation
     */
    public abstract Class<? extends P> getPresenterClass();

    protected abstract Dialog createDialog(Bundle savedInstanceState);

    /**
     * Find all views in root view
     */
    protected abstract void initView(View view);
}
