package com.humandevice.android.mvpframework;

import android.os.Bundle;


/**
 * @date 18.02.2016
 */
public class PresenterDelegate<V extends LayerView, P extends Presenter<V>> {
    private static final String PRESENTER_INDEX_KEY = "presenter-index";

    private P presenter;
    private long presenterId;
    private boolean destroyedBySystem;

    public void onCreate(PresenterCache cache, Bundle savedInstanceState, Class<? extends P> presenter) {
        if (savedInstanceState == null) {
            presenterId = cache.generateId();
        } else {
            presenterId = savedInstanceState.getLong(PRESENTER_INDEX_KEY);
        }
        this.presenter = cache.getPresenter(presenterId);
        if (this.presenter == null) {
            this.presenter = PresenterFactory.create(presenter);
            cache.setPresenter(presenterId, this.presenter);
        }
        Bundle bundle = PresenterBundleUtils.getPresenterBundle(savedInstanceState, presenterId);
        this.presenter.onCreate(bundle);
    }

    public void onViewCreated(V view) {
        presenter.onBindView(view);
    }

    public void onDestroyView() {
        presenter.onUnbindView();
    }

    public void onResume() {
        destroyedBySystem = false;
        presenter.onResume();
    }

    public void onPause() {
        presenter.onPause();
    }

    public void onSaveInstanceState(Bundle outState) {
        destroyedBySystem = true;
        outState.putLong(PRESENTER_INDEX_KEY, presenterId);
        Bundle bundle = new Bundle();
        presenter.onSaveInstanceState(bundle);
        PresenterBundleUtils.setPresenterBundle(outState, bundle, presenterId);
    }

    public void onDestroy(PresenterCache cache) {
        if (!destroyedBySystem) {
            // User is exiting this view, remove presenter from the cache
            cache.setPresenter(presenterId, null);
            presenter.onDestroy();
        }
    }

    public P getPresenter() {
        return presenter;
    }

}
