package com.humandevice.android.mvpframework;

import android.app.Activity;
import android.app.Fragment;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.CallSuper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;

/**
 * @date 18.02.2016
 */
public abstract class PresenterFragment<V extends LayerView, P extends Presenter<V>> extends Fragment implements LayerView {
	private PresenterCache presenterCache;
	private PresenterDelegate<V, P> presenterDelegate = new PresenterDelegate<>();
	
	/**
	 * Overridden version of M's getContext - works with previous versions.
	 */
	@Override
	public Context getContext() {
		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
			return getActivity();
		} else {
			return super.getContext();
		}
	}
	
	@Override
	public void onCreate(@Nullable Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		presenterDelegate.onCreate(presenterCache, savedInstanceState, getPresenterClass());
	}
	
	@Override
	public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
		super.onViewCreated(view, savedInstanceState);
		initView(view);
	}
	
	@Override
	public void onViewStateRestored(Bundle savedInstanceState) {
		super.onViewStateRestored(savedInstanceState);
		//noinspection unchecked
		presenterDelegate.onViewCreated((V) this);
	}
	
	@SuppressWarnings("deprecation")
	@Override
	public final void onAttach(Activity activity) {
		super.onAttach(activity);
		onFragmentAttach(activity);
	}
	
	@Override
	public final void onAttach(Context context) {
		super.onAttach(context);
		onFragmentAttach(context);
	}
	
	@CallSuper
	public void onFragmentAttach(Context context) {
		checkActivityClass(context);
	}
	
	@Override
	public void onResume() {
		super.onResume();
		presenterDelegate.onResume();
	}
	
	@Override
	public void onPause() {
		super.onPause();
		presenterDelegate.onPause();
	}
	
	@Override
	public void onSaveInstanceState(@NonNull Bundle outState) {
		super.onSaveInstanceState(outState);
		presenterDelegate.onSaveInstanceState(outState);
	}
	
	@Override
	public void onDestroyView() {
		super.onDestroyView();
		presenterDelegate.onDestroyView();
	}
	
	@Override
	public void onDestroy() {
		super.onDestroy();
		presenterDelegate.onDestroy(presenterCache);
	}
	
	
	private void checkActivityClass(Context context) {
		if (context instanceof PresenterCache) {
			presenterCache = (PresenterCache) context;
		} else {
			throw new RuntimeException(getClass().getSimpleName() + " must be attached to an Activity that implements "
					+ PresenterCache.class.getSimpleName());
		}
	}
	
	public P getPresenter() {
		return presenterDelegate.getPresenter();
	}
	
	/**
	 * Get Class of the presenter implementation
	 */
	public abstract Class<? extends P> getPresenterClass();
	
	/**
	 * Find all views in root view
	 */
	protected abstract void initView(View view);
	
}
