package com.humandevice.resttools.rest;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.humandevice.resttools.rest.exceptions.RefreshTokenException;
import com.humandevice.resttools.rest.exceptions.RequestException;
import com.humandevice.resttools.service.UserService;
import com.rafalzajfert.androidlogger.Logger;

import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.Credentials;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.Route;

/**
 * Klasa dostarczająca podstawowe metody komunikacji z api
 *
 * @author Rafal Zajfert
 * @date 07.01.2016
 */
public abstract class Request<T> {
    protected final ObjectReader mObjectReader;
    private final ObjectMapper mObjectMapper = new ObjectMapper();
    protected final ObjectWriter mObjectWriter = mObjectMapper.writer();
    protected OkHttpClient mHttpClient;

    protected static UserService mUserService;
    protected static RestConfig sConfig;

    private RequestExecutor mExecutor;
    private Map<String, Object> mBodyParameters = new HashMap<>();
    private Map<String, Object> mUrlParameters = new HashMap<>();
    private String[] mUrlSegments;
    private boolean mAddApiVersion = true;

    public static void setUserService(UserService userService) {
        mUserService = userService;
    }

    public static void setRestConfig(RestConfig credentials) {
        sConfig = credentials;
    }

    protected Request() {
        if (mUserService == null) {
            throw new IllegalArgumentException("No " + UserService.class.getSimpleName() + " passed. Use method setUserService(UserService) to deliver " +
                    UserService.class.getSimpleName() + ".");
        }
        if (sConfig == null) {
            throw new IllegalArgumentException("No " + RestConfig.class.getSimpleName() + " passed. Use method setUserService(UserService) to deliver" +
                    RestConfig.class.getSimpleName() + ".");
        }
        OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder();
        clientBuilder.connectionPool(new ConnectionPool(20, sConfig.ALIVE_DURATION, TimeUnit.MILLISECONDS));
        if (sConfig.API_BASIC_AUTH) {
            clientBuilder.authenticator(new Authenticator() {
                @Override
                public okhttp3.Request authenticate(Route route, Response response) throws IOException {
                    okhttp3.Request.Builder requestBuilder = response.request()
                            .newBuilder()
                            .header("Authorization", Credentials.basic(sConfig.API_BASIC_USER,
                                    sConfig.API_BASIC_PASSWORD));
                    return requestBuilder.build();
                }
            });
        }
        Interceptor logging = new Interceptor() {

            @Override
            public Response intercept(Chain chain) throws IOException {
                okhttp3.Request request = chain.request();

                long t1 = System.nanoTime();
                Logger.debug(String.format("Sending request %s on %s%n%s",
                        request.url(), chain.connection(), request.headers()));

                Response response = chain.proceed(request);

                long t2 = System.nanoTime();
                Logger.debug(String.format("Received response for %s in %.1fms%n%s",
                        response.request().url(), (t2 - t1) / 1e6d, response.headers()));

                return response;
            }
        };
        mHttpClient = clientBuilder
                .addInterceptor(logging)
                .connectTimeout(1, TimeUnit.MINUTES)
                .readTimeout(1, TimeUnit.MINUTES)
                .build();
        mExecutor = new RequestExecutor(20, sConfig.ALIVE_DURATION);
        mObjectMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mObjectMapper.registerModule(new SimpleModule());
        TypeReference<?> parameterType = getParameterType();
        if (parameterType != null) {
            mObjectReader = mObjectMapper.readerFor(parameterType);
        } else {
            mObjectReader = mObjectMapper.readerFor(getParameterClass());
        }
    }

    /**
     * Metoda zwracająca typ obiektu zwracanego, jeśli metoda nie została przeciążona to zwraca null
     */
    protected TypeReference<?> getParameterType() {
        return null;
    }

    /**
     * metoda zwracająca classę obiektu oczekiwanego w odpowiedzi
     */
    @SuppressWarnings("unchecked")
    protected Class<T> getParameterClass() {
        ParameterizedType genericSuperclass = (ParameterizedType) getClass().getGenericSuperclass();
        Type type = genericSuperclass.getActualTypeArguments()[0];
        if (type instanceof Class) {
            return (Class<T>) type;
        } else if (type instanceof ParameterizedType) {
            return (Class<T>) ((ParameterizedType) type).getRawType();
        } else {
            return null;
        }
    }

    /**
     * Metoda uruchamiająca wykonywanie zapytania
     */
    public RequestFuture<T> execute() {
        return execute(null);
    }

    /**
     * Metoda uruchamiająca wykonywanie zapytania, po zakończeniu uruchamiana jest metoda listenera
     */
    public RequestFuture<T> execute(RequestListener<T> listener) {
        return mExecutor.submit(createRequestTask(), listener);
    }

    /**
     * Utworzenie nowego zadania dla zapytania
     *
     * @return callable zapytania
     */
    @NonNull
    protected Callable<T> createRequestTask() {
        return new Callable<T>() {
            @Override
            public T call() throws Exception {
                long executionStartTime = System.nanoTime();
                if (isTokenRequest()) {
                    checkTokenValidity();
                }
                prepareRequest();
                Response response = request();
                if (response == null) {
                    return null;
                }

                int status = response.code();
                String content = response.body().string();
                Logger.verboseF("Request execution took: %.3fms", (float) (System.nanoTime() - executionStartTime) / 1_000_000);
                Logger.verbose(content);
                if (status == 200) {
                    return mObjectReader.readValue(content);
                } else {
                    throw new RequestException(content, status);
                }
            }
        };
    }

    /**
     * Sprawdzenie czy zapytanie wymaga autoryzacji przy użyciu tokena
     */
    private boolean isTokenRequest() {
        return Request.this instanceof TokenRequest;
    }

    /**
     * Metoda jest wykonywana dla Requestów implementujących interfejs TokenRequest.
     * Pierwszą rzeczą jest sprawdzanie zalogowanie użytkownika, gdy nie jest - pomijamy wykonywanie dalszej części
     * kody. Następie używany jest ThreadLock - singleton synchronizujący dostęp wsyzstkich wątków z niego korzystających.
     * Dzięki ThreadLock tylko jeden wątek odświeży token, zapisze go i udostępni innym wątkom do użycia.
     */
    @WorkerThread
    private void checkTokenValidity() throws RefreshTokenException {
        if (!mUserService.isLogged()) {
            return;
        }
        final ThreadLock lock = ThreadLock.getInstance();
        try {
            lock.waitIfLocked();
        } catch (InterruptedException e) {
            throw new RefreshTokenException("Locked thread interrupted!", e);
        }
        if (mUserService.isTokenValid()) {
            return;
        }
        try {
            lock.lock();
        } catch (InterruptedException e) {
            throw new RefreshTokenException("Locked thread interrupted!", e);
        }
        try {
            mUserService.refreshToken();
        } catch (Exception e) {
            mUserService.logout();
            throw new RefreshTokenException("Problem during obtaining refresh token", e);
        } finally {
            lock.unlock();
        }
    }

    /**
     * Metoda zwracająca wynik zapytania api
     */
    @WorkerThread
    protected abstract Response request() throws IOException;

    /**
     * Metoda, w której powinna byc wykonana konfiguracja zapytania.
     *
     * @see #setUrlSegments(String...)
     * @see #putParameter(String, Object)
     * @see #putUrlParameter(String, Object)
     */
    protected abstract void prepareRequest();

    /**
     * Metoda zwraca kompletny adres url do metody api. Jeśli do adresu mają zostać dodane parametry to musi to być wykonane przed
     * wywołaniem tej funkcji.
     *
     * @see #putUrlParameter(String, Object)
     */
    protected HttpUrl getUrl() {
        if (mUrlSegments == null) {
            throw new IllegalStateException("Set url segments in " + this.getClass().getSimpleName() + " in prepareRequest method");
        }

        HttpUrl.Builder builder = new HttpUrl.Builder()
                .scheme(sConfig.API_SCHEME)
                .port(sConfig.API_PORT)
                .host(sConfig.API_HOST);
        if (sConfig.API_BASIC_AUTH) {
            builder.username(sConfig.API_BASIC_USER)
                    .password(sConfig.API_BASIC_PASSWORD);
        }
        if (mAddApiVersion) {
            builder.addPathSegment(sConfig.API_VERSION);
        }
        for (String segment : mUrlSegments) {
            builder.addPathSegment(segment);
        }
        if (mUrlParameters != null && mUrlParameters.size() > 0) {
            for (Map.Entry<String, Object> param : mUrlParameters.entrySet()) {
                builder.addQueryParameter(param.getKey(), param.getValue() + "");
            }
        }
        if (isTokenRequest()) {
            builder.addQueryParameter("access_token", mUserService.getToken().getToken());
        }
        return builder.build();
    }

    /**
     * Ustawienie adresu api
     *
     * @param urlSegments elementy ścieżki url do odpowiedniej metody api np.: <br/>
     *                    dla adresu http://example.com/v1/get/user metoda
     *                    powinna być wywołana: {@code getUrlBuilder("get", "user");}
     */
    protected void setUrlSegments(String... urlSegments) {
        mUrlSegments = urlSegments;
    }

    /**
     * Dodanie parametru do body zapytania
     */
    protected void putParameter(@NonNull String parameterName, @Nullable Object value) {
        putParameter(null, parameterName, value);
    }

    /**
     * Dodanie parametru do body zapytania
     */
    protected void putParameter(@Nullable String objectName, @NonNull String parameterName, @Nullable Object value) {
        if (value instanceof Boolean) {
            value = !((Boolean) value) ? 0 : 1;
        }
        if (TextUtils.isEmpty(objectName)) {
            mBodyParameters.put(parameterName, value);
        } else if (mBodyParameters.containsKey(objectName)) {
            //noinspection unchecked
            Map<String, Object> objectMap = (HashMap<String, Object>) mBodyParameters.get(objectName);
            objectMap.put(parameterName, value);
        } else {
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put(parameterName, value);
            mBodyParameters.put(objectName, objectMap);
        }
    }

    /**
     * Dodanie parametru do adresu url.
     */
    protected void putUrlParameter(@NonNull String parameterName, @Nullable Object value) {
        putUrlParameter(null, parameterName, value);
    }

    /**
     * Dodanie parametru do adresu url
     *
     * @param objectName nazwa obiektu do jakiego ma zostać dopisany parametr, wynikowa nazwą będzie: objectName[parameterName]
     */
    protected void putUrlParameter(@Nullable String objectName, @NonNull String parameterName, @Nullable Object value) {
        if (value instanceof Boolean) {
            value = !((Boolean) value) ? 0 : 1;
        }
        mUrlParameters.put(getParamName(objectName, parameterName), value);
    }

    /**
     * Metoda sprawdza czy parametr wymaga uzupełnienia o nazwę formy, jeśli tak to jest ona dodawana
     *
     * @param objectName nazwa obiektu do którego dołączany jest parametr
     * @param param      nazwa parametru
     * @return nazwa parametru z nazwą formy jeśli jest wymagana
     */
    private String getParamName(String objectName, String param) {
        if (TextUtils.isEmpty(objectName)) {
            return param;
        } else {
            return objectName + '[' + param + ']';
        }
    }

    /**
     * wywołanie tej metody powoduje wyłączenie dodawania wersji api ({@link RestConfig#API_VERSION}) do adresu url zapytania
     */
    protected void withoutApiVersion() {
        mAddApiVersion = false;
    }

    /**
     * metoda zwracająca body zapytania zawierające {@code parameters}
     *
     * @return body zapytania lub null jeśli nie zostały przesłane żadne parametry
     */
    @Nullable
    protected RequestBody getRequestBody() throws JsonProcessingException {
        RequestBody body = null;
        String parameters = getBodyParametersJson();
        if (!TextUtils.isEmpty(parameters)) {
            Logger.info(parameters);
            body = RequestBody.create(sConfig.MEDIA_TYPE, parameters);
        }
        return body;
    }

    /**
     * Metoda zwracająca parametry zapytania w formie jsona
     *
     * @return json z parametrami zapytania, null jeśli nie ma żadnego parametru
     * @throws JsonProcessingException jeśli nie można zmapować modelu do jsona
     */
    @NonNull
    private String getBodyParametersJson() throws JsonProcessingException {
        return mObjectWriter.writeValueAsString(mBodyParameters);
    }

}
