package com.humandevice.resttools.rest.exceptions;

import android.support.annotation.NonNull;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

/**
 * Exception rzucany w przypadku prawidłowej odpowiedzi z serwera zawierającej informacje o błędzie
 *
 * @author Rafal Zajfert
 * @date 29.02.2016
 */
@SuppressWarnings("unused")
public class RequestException  extends ExecutionException {

    /**
     * Request accepted but data is incomplete / invalid
     */
    public static final int CODE_ACCEPTED = 202;

    /**
     * The request was invalid. You may be missing a required argument or provided bad data. An error message will be returned explaining what happened.
     */
    public static final int CODE_BAD_REQUEST = 400;
    /**
     * The authentication you provided is invalid.
     */
    public static final int CODE_UNAUTHORIZED = 401;
    /**
     * You don't have permission to complete the operation or access the resource.
     */
    public static final int CODE_FORBIDDEN = 403;
    /**
     * You requested an invalid method.
     */
    public static final int CODE_NOT_FOUND = 404;
    /**
     * The method specified in the Request-Line is not allowed for the resource identified by the Request-URI. (used POST instead of PUT)
     */
    public static final int CODE_METHOD_NOT_ALLOWED = 405;
    /**
     * The server timed out waiting for the request.
     */
    public static final int CODE_TIMEOUT = 408;
    /**
     * You have exceeded the rate limit.
     */
    public static final int CODE_TOO_MANY_REQUESTS = 429;
    /**
     * Something is wrong on our end. We'll investigate what happened.
     */
    public static final int CODE_INTERNAL_SERVER_ERROR = 500;
    /**
     * The method you requested is currently unavailable (due to maintenance or high load).
     */
    public static final int CODE_SERVICE_UNAVAILABLE = 503;
    /**
     * Non server exception
     */
    public static final int UNKNOWN = -1;
    /**
     * Waiting thread is activated before the condition it was waiting for has been satisfied.
     */
    public static final int INTERRUPTED = -2;
    /**
     * Blocking operation times out
     */
    public static final int TIMEOUT = -3;

    @NonNull
    protected ErrorResponse mErrorResponse;

    public RequestException(@NonNull Exception e) {
        super(e);
        mErrorResponse = new ErrorResponse();
        mErrorResponse.setName("Unknown Error");
        mErrorResponse.setMessage(e.getMessage());
        mErrorResponse.setCode(UNKNOWN);
        if (e instanceof InterruptedException){
            mErrorResponse.setStatusCode(INTERRUPTED);
        } else if (e instanceof TimeoutException){
            mErrorResponse.setStatusCode(TIMEOUT);
        } else {
            mErrorResponse.setStatusCode(UNKNOWN);
        }
        mErrorResponse.setErrors(new HashMap<String, String[]>());
    }

    /**
     * Konstruktor wyjątku parsujący odpowiedź z serwera
     *
     * @param errorJson
     * @param requestCode
     * @throws IOException
     */
    public RequestException(@NonNull String errorJson, int requestCode) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        ObjectReader errorReader = objectMapper.readerFor(ErrorResponse.class);
        mErrorResponse = errorReader.readValue(errorJson);
        mErrorResponse.setStatusCode(requestCode);
    }

    /**
     * Metoda sprawdzająca czy exception został wywołany na etapie połączenia z serwerem
     * @return
     */
    public boolean isConnectionError(){
        return mErrorResponse.statusCode <= 0;
    }

    /**
     * Metoda sprawdzająca czy exception został wywołany na serwerze i został zwrócony json błędu
     * @return
     */
    public boolean isResponseError(){
        return mErrorResponse.statusCode > 0;
    }

    /**
     * Wiadomość błędu
     */
    @Override
    public String getMessage() {
        return mErrorResponse.message;
    }

    /**
     * Kod błędu
     */
    public int getCode() {
        return mErrorResponse.code;
    }

    /**
     * Kod odpowiedzi
     */
    public int getStatusCode() {
        return mErrorResponse.statusCode;
    }

    /**
     * błędy w formularzu
     * @return mapa błędów gdzie klucz to nazwa pola, a wartość to błędy
     */
    public Map<String, String[]> getErrors() {
        return mErrorResponse.errors;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ErrorResponse {

        /**
         * Nazwa błędu
         */
        private String name;

        /**
         * Wiadomość dla użytkownika
         */
        private String message;

        /**
         * Kod błędu
         */
        private int code;

        /**
         * Kod odpowiedzi
         */
        @JsonProperty("status")
        private int statusCode;

        private Map<String, String[]> errors = new HashMap<>();

        public void setMessage(String message) {
            this.message = message;
        }

        public void setCode(int code) {
            this.code = code;
        }

        public void setStatusCode(int statusCode) {
            this.statusCode = statusCode;
        }

        public void setErrors(Map<String, String[]> errors) {
            this.errors = errors;
        }

        public void setName(String name) {
            this.name = name;
        }
    }
}
