package com.humandevice.android.resttools.rest;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.humandevice.android.resttools.rest.exceptions.ConnectionRestrictionFailedException;
import com.humandevice.android.resttools.rest.exceptions.RefreshTokenException;
import com.humandevice.android.resttools.rest.exceptions.RequestException;
import com.humandevice.android.resttools.service.UserService;
import com.rafalzajfert.androidlogger.Logger;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.Route;

/**
 * Klasa dostarczająca podstawowe metody komunikacji z api
 *
 * @author Rafal Zajfert
 * @date 07.01.2016
 */
@SuppressWarnings("unused")
public abstract class Request<T> {
    protected static final String CONTENT_TYPE = "Content-Type";
    protected static final MediaType MEDIA_TYPE = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");
    protected static final MediaType MULTIPART_MEDIA_TYPE = MultipartBody.FORM;
    protected static final String ACCESS_TOKEN = "access_token";
    protected static final String AUTHORIZATION = "Authorization";

    protected static UserService mUserService;
    protected static RestConfiguration sConfiguration;

    protected OkHttpClient mHttpClient;
    private RequestExecutor mExecutor;

    private String[] mUrlSegments;
    private Map<String, Object> mBodyParameters = new HashMap<>();
    private Map<String, Object> mUrlParameters = new HashMap<>();
    protected boolean isMultipartRequest;

    protected Request() {
        if (mUserService == null) {
            throw new IllegalArgumentException(
                    String.format(Locale.getDefault(), "No %1$s  passed. Use method setUserService(UserService) to deliver %1$s.",
                            UserService.class.getSimpleName()));
        }
        if (sConfiguration == null) {
            throw new IllegalArgumentException(
                    String.format(Locale.getDefault(), "No %1$s  passed. Use method setRestConfiguration(RestConfiguration) to deliver %1$s.",
                            RestConfiguration.class.getSimpleName()));
        }

        createHttpClient();
        mExecutor = new RequestExecutor(sConfiguration.mMaxConnections, sConfiguration.mTimeout);

    }

    private void createHttpClient() {
        OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder()
                .connectionPool(new ConnectionPool(sConfiguration.mMaxConnections, sConfiguration.mTimeout, TimeUnit.MILLISECONDS))
                .connectTimeout(sConfiguration.mTimeout, TimeUnit.MILLISECONDS)
                .readTimeout(sConfiguration.mTimeout, TimeUnit.MILLISECONDS);

        if (!TextUtils.isEmpty(sConfiguration.mAuthorization)) {
            clientBuilder.authenticator(new Authenticator() {
                @Override
                public okhttp3.Request authenticate(Route route, Response response) throws IOException {
                    okhttp3.Request.Builder requestBuilder = response.request()
                            .newBuilder()
                            .header(AUTHORIZATION, sConfiguration.mAuthorization);
                    return requestBuilder.build();
                }
            });
        }
        mHttpClient = clientBuilder.build();
    }

    public static void setUserService(UserService userService) {
        mUserService = userService;
    }

    public static void setRestConfiguration(RestConfiguration credentials) {
        sConfiguration = credentials;
    }

    /**
     * Metoda uruchamiająca wykonywanie zapytania
     */
    public RequestFuture<T> execute() {
        return execute(null);
    }

    /**
     * Metoda uruchamiająca wykonywanie zapytania, po zakończeniu uruchamiana jest metoda listenera
     */
    public RequestFuture<T> execute(RequestListener<T> listener) {
        return mExecutor.submit(createRequestTask(), listener);
    }

    /**
     * Utworzenie nowego zadania dla zapytania
     *
     * @return callable zapytania
     */
    @NonNull
    protected Callable<T> createRequestTask() {
        return new Callable<T>() {
            @Override
            public T call() throws Exception {
                try {
                    checkConnectionRestrictions();
                } catch (Exception e) {
                    throw new RequestException(e);
                }
                long executionStartTime = System.nanoTime();
                if (isTokenRequest()) {
                    checkTokenValidity();
                }
                prepareRequest();
                Response response = request();
                if (response == null) {
                    return null;
                }

                int status = response.code();
                String content = response.body().string();
                Logger.verboseF("Request execution took: %.3fms", (float) (System.nanoTime() - executionStartTime) / 1_000_000);
                Logger.verbose(content);
                if (status == 200) {
                    return getObjectReader().readValue(content);
                } else {
                    throw new RequestException(content, status);
                }
            }
        };
    }

    private void checkConnectionRestrictions() throws ConnectionRestrictionFailedException {
        if (sConfiguration.mConnectionRestriction != null){
            sConfiguration.mConnectionRestriction.onCheckRestriction();
        }
    }

    public interface ConnectionRestriction {
        void onCheckRestriction() throws ConnectionRestrictionFailedException;
    }

    /**
     * Utworzenie nowego readera obiektów
     */
    private ObjectReader getObjectReader() {
        TypeReference<?> parameterType = getParameterType();
        if (parameterType != null) {
            return JacksonMapper.getInstance().getReaderFor(parameterType);
        } else {
            return JacksonMapper.getInstance().getReaderFor(getParameterClass());
        }
    }

    /**
     * Metoda zwracająca typ obiektu zwracanego, jeśli metoda nie została przeciążona to zwraca null
     */
    protected TypeReference<?> getParameterType() {
        return null;
    }

    /**
     * metoda zwracająca classę obiektu oczekiwanego w odpowiedzi
     */
    @SuppressWarnings("unchecked")
    protected Class<T> getParameterClass() {
        ParameterizedType genericSuperclass = (ParameterizedType) getClass().getGenericSuperclass();
        Type type = genericSuperclass.getActualTypeArguments()[0];
        if (type instanceof Class) {
            return (Class<T>) type;
        } else if (type instanceof ParameterizedType) {
            return (Class<T>) ((ParameterizedType) type).getRawType();
        } else {
            return null;
        }
    }

    /**
     * Sprawdzenie czy zapytanie wymaga autoryzacji przy użyciu tokena
     */
    private boolean isTokenRequest() {
        return Request.this instanceof TokenRequest;
    }

    /**
     * Metoda jest wykonywana dla Requestów implementujących interfejs TokenRequest.
     * Pierwszą rzeczą jest sprawdzanie zalogowanie użytkownika, gdy nie jest - pomijamy wykonywanie dalszej części
     * kody. Następie używany jest ThreadLock - singleton synchronizujący dostęp wsyzstkich wątków z niego korzystających.
     * Dzięki ThreadLock tylko jeden wątek odświeży token, zapisze go i udostępni innym wątkom do użycia.
     */
    @WorkerThread
    private void checkTokenValidity() throws RefreshTokenException {
        if (!mUserService.isLogged()) {
            return;
        }
        final ThreadLock lock = ThreadLock.getInstance();
        try {
            lock.waitIfLocked();
        } catch (InterruptedException e) {
            throw new RefreshTokenException("Locked thread interrupted!", e);
        }
        if (mUserService.isTokenValid()) {
            return;
        }
        try {
            lock.lock();
        } catch (InterruptedException e) {
            throw new RefreshTokenException("Locked thread interrupted!", e);
        }
        try {
            mUserService.refreshToken();
        } catch (Exception e) {
            mUserService.logout();
            throw new RefreshTokenException("Problem during obtaining refresh token", e);
        } finally {
            lock.unlock();
        }
    }

    /**
     * Metoda, w której powinna byc wykonana konfiguracja zapytania.
     *
     * @see #setUrlSegments(String...)
     * @see #putParameter(String, Object)
     * @see #putUrlParameter(String, Object)
     */
    protected abstract void prepareRequest();

    /**
     * Metoda zwracająca wynik zapytania api
     */
    @WorkerThread
    protected abstract Response request() throws IOException;

    /**
     * Metoda zwraca kompletny adres url do metody api. Jeśli do adresu mają zostać dodane parametry to musi to być wykonane przed
     * wywołaniem tej funkcji.
     *
     * @see #putUrlParameter(String, Object)
     */
    protected HttpUrl getUrl() {
        if (mUrlSegments == null) {
            throw new IllegalStateException("Set url segments in " + this.getClass().getSimpleName() + " in prepareRequest method");
        }

        HttpUrl.Builder builder = new HttpUrl.Builder()
                .scheme(sConfiguration.mApiScheme)
                .port(sConfiguration.mApiPort)
                .host(sConfiguration.mApiHost);
        if (!TextUtils.isEmpty(sConfiguration.mAuthorization)) {
            builder.username(sConfiguration.mAuthorizationUser)
                    .password(sConfiguration.mAuthorizationPassword);
        }
        for (String segment : mUrlSegments) {
            builder.addPathSegment(segment);
        }
        if (mUrlParameters != null && mUrlParameters.size() > 0) {
            for (Entry<String, Object> param : mUrlParameters.entrySet()) {
                builder.addQueryParameter(param.getKey(), param.getValue() + "");
            }
        }
        if (isTokenRequest()) {
            builder.addQueryParameter(ACCESS_TOKEN, mUserService.getToken().getToken());
        }
        return builder.build();
    }

    /**
     * Ustawienie adresu api
     *
     * @param urlSegments elementy ścieżki url do odpowiedniej metody api np.: <br/>
     *                    dla adresu http://example.com/v1/get/user metoda
     *                    powinna być wywołana: {@code getUrlBuilder("get", "user");}
     */
    protected void setUrlSegments(String... urlSegments) {
        mUrlSegments = urlSegments;
    }

    /**
     * Dodanie parametru do adresu url.
     */
    protected void putUrlParameter(@NonNull String parameterName, @Nullable Object value) {
        putUrlParameter(null, parameterName, value);
    }

    /**
     * Dodanie parametru do adresu url
     *
     * @param objectName nazwa obiektu do jakiego ma zostać dopisany parametr, wynikowa nazwą będzie: objectName[parameterName]
     */
    protected void putUrlParameter(@Nullable String objectName, @NonNull String parameterName, @Nullable Object value) {
        if (sConfiguration.mCastBooleanToInt && value instanceof Boolean) {
            value = !((Boolean) value) ? 0 : 1;
        }
        mUrlParameters.put(getFormParameterName(objectName, parameterName), value);
    }

    /**
     * Metoda sprawdza czy parametr wymaga uzupełnienia o nazwę formy, jeśli tak to jest ona dodawana
     *
     * @param formName nazwa obiektu do którego dołączany jest parametr
     * @param param    nazwa parametru
     * @return nazwa parametru z nazwą formy jeśli jest wymagana
     */
    private String getFormParameterName(String formName, String param) {
        if (TextUtils.isEmpty(formName)) {
            return param;
        } else {
            return formName + '[' + param + ']';
        }
    }

    /**
     * Dodanie parametru do body zapytania
     */
    protected void putParameter(@NonNull String parameterName, @Nullable Object value) {
        putParameter(null, parameterName, value);
    }

    /**
     * Dodanie parametru do body zapytania
     */
    protected void putParameter(@Nullable String objectName, @NonNull String parameterName, @Nullable Object value) {
        if (sConfiguration.mCastBooleanToInt && value instanceof Boolean) {
            value = !((Boolean) value) ? 0 : 1;
        }
        if (value instanceof File) {
            isMultipartRequest = true;
        }
        if (TextUtils.isEmpty(objectName)) {
            mBodyParameters.put(parameterName, value);
        } else if (mBodyParameters.containsKey(objectName)) {
            //noinspection unchecked
            Map<String, Object> objectMap = (HashMap<String, Object>) mBodyParameters.get(objectName);
            objectMap.put(parameterName, value);
        } else {
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put(parameterName, value);
            mBodyParameters.put(objectName, objectMap);
        }
    }

    /**
     * metoda zwracająca body zapytania zawierające {@code parameters}
     *
     * @return body zapytania lub null jeśli nie zostały przesłane żadne parametry
     */
    @Nullable
    protected RequestBody getRequestBody() throws JsonProcessingException {
        RequestBody body = null;
        if (isMultipartRequest) {
            MultipartBody.Builder bodyBuilder = new MultipartBody.Builder();
            bodyBuilder.setType(MULTIPART_MEDIA_TYPE);
//
            Iterator<Entry<String, Object>> parametersIterator = mBodyParameters.entrySet().iterator();
            while (parametersIterator.hasNext()) {
                Entry<String, Object> param = parametersIterator.next();
                Object value = param.getValue();
                Logger.error(param.getKey(), value.getClass().getSimpleName());
                if (value instanceof Map) {
                    //noinspection unchecked
                    Iterator<Entry<String, Object>> iterator = ((Map<String, Object>) value).entrySet().iterator();
                    while (iterator.hasNext()) {
                        Entry<String, Object> object = iterator.next();
                        if (object instanceof File) {
                            bodyBuilder.addFormDataPart(param.getKey() + '[' + object.getKey() + ']', ((File) value).getName(),
                                    RequestBody.create(MediaType.parse("image/png"), (File) value));
                            iterator.remove();
                        }
                    }
                    if (((Map) value).isEmpty()) {
                        parametersIterator.remove();
                    }
                } else if (value instanceof File) {
                    Logger.error(param.getKey(), ((File) value).getName());
                    bodyBuilder.addFormDataPart(param.getKey(), ((File) value).getName(),
                            RequestBody.create(MediaType.parse("image/png"), (File) value));
                    parametersIterator.remove();
                }
            }

            for (Entry<String, Object> object : mBodyParameters.entrySet()) {
                if (object.getValue() instanceof Map) {
                    //noinspection unchecked
                    for (Entry<String, Object> field : ((Map<String, Object>) object.getValue()).entrySet()) {
                        bodyBuilder.addFormDataPart(getFormParameterName(object.getKey(), field.getKey()), String.valueOf(field.getValue()));
                    }
                } else {
                    bodyBuilder.addFormDataPart(object.getKey(), String.valueOf(object.getValue()));
                }
            }
            body = bodyBuilder.build();
        } else {
            FormBody.Builder bodyBuilder = new FormBody.Builder();
            for (Entry<String, Object> object : mBodyParameters.entrySet()) {
                if (object.getValue() instanceof Map) {
                    //noinspection unchecked
                    for (Entry<String, Object> field : ((Map<String, Object>) object.getValue()).entrySet()) {
                        bodyBuilder.add(getFormParameterName(object.getKey(), field.getKey()), String.valueOf(field.getValue()));
                    }
                } else {
                    bodyBuilder.add(object.getKey(), String.valueOf(object.getValue()));
                }
            }
            body = bodyBuilder.build();
        }
        return body;
    }

    /**
     * Metoda zwracająca parametry zapytania w zakodowaniej do wysłania
     */
    @NonNull
    private String getBodyParametersString() throws JsonProcessingException {
        return JacksonMapper.getInstance().getObjectWriter().writeValueAsString(mBodyParameters);
    }
}
