package com.humandevice.android.resttools.rest;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.humandevice.android.resttools.rest.exceptions.RequestException;
import com.humandevice.android.resttools.rest.listeners.RequestListener;
import com.humandevice.android.resttools.rest.listeners.RequestPoolListener;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * TODO Dokumentacja
 *
 * @author Rafal Zajfert
 * @date 2016-07-07
 */
public class RequestPool {

	protected Map<Integer, Request> mRequestPool = new LinkedHashMap<>();
	protected Map<Integer, PostTask> mPostTaskMap = new LinkedHashMap<>();

	protected boolean mExecuted;
	protected boolean mStopIfFail = false;

	protected RequestExecutor mExecutor;
	protected RequestPoolListener mListener;
	protected Map<Integer, RequestFuture> mDoneTasks = new LinkedHashMap<>();

	private Iterator<Map.Entry<Integer, Request>> mExecuteIterator;
	private Map.Entry<Integer, Request> mCurrentExecutedTask;

	public RequestPool() {
		this(1);
	}

	protected RequestPool(int poolSize){
		mExecutor = new RequestExecutor(poolSize, 0L);
	}

	public <T> RequestPool addTask(@NonNull Request<T> request, int requestCode) {
		return addTask(request, requestCode, null);
	}

	public <T> RequestPool addTask(@NonNull Request<T> request, int requestCode, @Nullable PostTask<T> postTask) {
		if (mExecuted) {
			throw new IllegalStateException("New task cannot be added to the pool after executing.");
		}
		if (mRequestPool.containsKey(requestCode)) {
			throw new IllegalArgumentException("Task with this requestCode (" + requestCode + ")is already added.");
		}
		mRequestPool.put(requestCode, request);
		if (postTask != null) {
			mPostTaskMap.put(requestCode, postTask);
		}
		return this;
	}

	public RequestPool setStopIfFail() {
		mStopIfFail = true;
		return this;
	}

	public void execute(RequestPoolListener listener) {
		if (mExecuted) {
			throw new IllegalStateException("Already executed.");
		}
		mExecuted = true;
		mExecuteIterator = mRequestPool.entrySet().iterator();
		mListener = listener;

		executeNext();
	}

	private void executeNext() {
		if (mExecuteIterator.hasNext()) {
			mCurrentExecutedTask = mExecuteIterator.next();
			mExecutor.submit(mCurrentExecutedTask.getValue().createRequestTask(), mRequestListener);
		} else {
			mExecutor.shutdown();
			if (mListener != null) {
				try {
					mListener.done(mDoneTasks);
				} catch (RequestException ignored) {
					//RequestPoolListener.fail() will be thrown
				}

			}
		}
	}

	private RequestListener mRequestListener = new RequestListener() {
		@Override
		public void done(@NonNull RequestFuture task) {
			Integer requestCode = mCurrentExecutedTask.getKey();
			try {
				task.get();
			} catch (RequestException e) {
				if (mListener != null) {
					if (mListener.fail(e, requestCode)) {
						return;
					}
				}
				if (mStopIfFail) {
					return;
				}
			}
			mDoneTasks.put(requestCode, task);
			if (mPostTaskMap.containsKey(requestCode)) {
				try {
					//noinspection unchecked
					mPostTaskMap.get(requestCode).onPost(task);
				} catch (RequestException ignored) {
					//RequestPoolListener.fail() will be thrown
				}
			}
			executeNext();
		}
	};


	public interface PostTask<T> {
		void onPost(RequestFuture<T> task) throws RequestException;
	}
}
