package com.humandevice.android.resttools.rest;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.humandevice.android.resttools.adapters.HttpParameters;
import com.humandevice.android.resttools.adapters.JacksonMapper;
import com.humandevice.android.resttools.adapters.ReflectHttpUrlEncodedSerializer;
import com.humandevice.android.resttools.rest.exceptions.ConnectionRestrictionFailedException;
import com.humandevice.android.resttools.rest.exceptions.RefreshTokenException;
import com.humandevice.android.resttools.rest.exceptions.RequestException;
import com.humandevice.android.resttools.rest.listeners.RequestListener;
import com.humandevice.android.resttools.service.IUserService;
import com.rafalzajfert.androidlogger.Logger;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map.Entry;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.Route;

/**
 * Klasa dostarczająca podstawowe metody komunikacji z api
 *
 * @author Rafal Zajfert
 */
@SuppressWarnings({"unused", "WeakerAccess"})
public abstract class Request<T> {
	protected static final String CONTENT_TYPE = "Content-Type";
	protected static final MediaType MEDIA_TYPE = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");
	protected static final MediaType MULTIPART_MEDIA_TYPE = MultipartBody.FORM;
	protected static final String ACCESS_TOKEN = "access_token";
	protected static final String AUTHORIZATION = "Authorization";

	protected static RestConfiguration sConfiguration;

	protected OkHttpClient mHttpClient;
	private RequestExecutor mExecutor;

	private String[] mUrlSegments;
	private HttpParameters mBodyParameters = new HttpParameters();
	private HttpParameters mUrlParameters = new HttpParameters();
	protected boolean isMultipartRequest;
	private long mExecutionStartTime;
	private IUserService mUserService;
	private static ThreadLock sThreadLock = new ThreadLock();

	protected Request() {
		if (sConfiguration == null) {
			throw new IllegalArgumentException(
					String.format(Locale.getDefault(), "No %1$s  passed. Use method setRestConfiguration(RestConfiguration) to deliver %1$s.",
							RestConfiguration.class.getSimpleName()));
		}
		JacksonMapper.getInstance().setUnixTimeDeserializer(sConfiguration.isUnixTime());

		createHttpClient();
		mExecutor = new RequestExecutor(1, sConfiguration.mTimeout);

	}

	private void createHttpClient() {
		OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder()
				.connectionPool(new ConnectionPool(1, sConfiguration.mTimeout, TimeUnit.MILLISECONDS))
				.connectTimeout(sConfiguration.mTimeout, TimeUnit.MILLISECONDS)
				.readTimeout(sConfiguration.mTimeout, TimeUnit.MILLISECONDS);

		if (!TextUtils.isEmpty(sConfiguration.mAuthorization)) {
			clientBuilder.authenticator(new Authenticator() {
				@Override
				public okhttp3.Request authenticate(Route route, Response response) throws IOException {
					okhttp3.Request.Builder requestBuilder = response.request()
							.newBuilder()
							.header(AUTHORIZATION, sConfiguration.mAuthorization);
					return requestBuilder.build();
				}
			});
		}
		mHttpClient = clientBuilder.build();
	}

	public static void setRestConfiguration(RestConfiguration credentials) {
		sConfiguration = credentials;
	}

	/**
	 * Metoda uruchamiająca wykonywanie zapytania
	 */
	public RequestFuture<T> execute() {
		return execute(null);
	}

	/**
	 * Metoda uruchamiająca wykonywanie zapytania, po zakończeniu uruchamiana jest metoda listenera
	 */
	public RequestFuture<T> execute(RequestListener<T> listener) {
		Logger.verbose("execute:", getClass().getSimpleName());
		RequestFuture<T> future = mExecutor.submit(createRequestTask(), listener);
		mExecutor.shutdown();
		return future;
	}

	/**
	 * Utworzenie nowego zadania dla zapytania
	 *
	 * @return callable zapytania
	 */
	@NonNull
	protected Callable<T> createRequestTask() {
		return new Callable<T>() {
			@Override
			public T call() throws Exception {
				try {
					checkConnectionRestrictions(Request.this);
				} catch (Exception e) {
					throw new RequestException(e);
				}
				mExecutionStartTime = System.nanoTime();
				if (isTokenRequest()) {
					checkTokenValidity();
				}
				prepareRequest();
				Response response = request();
				if (response == null) {
					return null;
				}

				return readResponse(response);
			}
		};
	}

	protected T readResponse(Response response) throws Exception {
		int status = response.code();
		String content = response.body().string();
		response.close();
		Logger.verboseF("Request execution took: %.3fms", (float) (System.nanoTime() - mExecutionStartTime) / 1_000_000);
		Logger.verbose(content);
		if (sConfiguration.mSuccessStatusCodes.contains(status)) {
			return parseToObject(content);
		} else {
			throw new RequestException(content, status);
		}
	}

	protected T parseToObject(String content) throws Exception {
		return getObjectReader().readValue(content);
	}

	protected void checkConnectionRestrictions(Request<T> request) throws ConnectionRestrictionFailedException {
		if (sConfiguration.mConnectionRestriction != null) {
			sConfiguration.mConnectionRestriction.onCheckRestriction(request);
		}
	}

	public interface ConnectionRestriction {
		void onCheckRestriction(Request<?> request) throws ConnectionRestrictionFailedException;
	}

	/**
	 * Utworzenie nowego readera obiektów
	 */
	private ObjectReader getObjectReader() {
		TypeReference<?> parameterType = getParameterType();
		if (parameterType != null) {
			return JacksonMapper.getInstance().getReaderFor(parameterType);
		} else {
			return JacksonMapper.getInstance().getReaderFor(getParameterClass());
		}
	}

	/**
	 * Metoda zwracająca typ obiektu zwracanego, jeśli metoda nie została przeciążona to zwraca null
	 */
	protected TypeReference<?> getParameterType() {
		return null;
	}

	/**
	 * metoda zwracająca classę obiektu oczekiwanego w odpowiedzi
	 */
	@SuppressWarnings("unchecked")
	protected Class<T> getParameterClass() {
		ParameterizedType genericSuperclass = (ParameterizedType) getClass().getGenericSuperclass();
		Type type = genericSuperclass.getActualTypeArguments()[0];
		if (type instanceof Class) {
			return (Class<T>) type;
		} else if (type instanceof ParameterizedType) {
			return (Class<T>) ((ParameterizedType) type).getRawType();
		} else {
			return null;
		}
	}

	/**
	 * Sprawdzenie czy zapytanie wymaga autoryzacji przy użyciu tokena
	 */
	protected boolean isTokenRequest() {
		return Request.this instanceof TokenRequest;
	}

	/**
	 * Metoda jest wykonywana dla Requestów implementujących interfejs TokenRequest.
	 * Pierwszą rzeczą jest sprawdzanie zalogowanie użytkownika, gdy nie jest - pomijamy wykonywanie dalszej części
	 * kody. Następie używany jest ThreadLock - singleton synchronizujący dostęp wsyzstkich wątków z niego korzystających.
	 * Dzięki ThreadLock tylko jeden wątek odświeży token, zapisze go i udostępni innym wątkom do użycia.
	 */
	@WorkerThread
	protected void checkTokenValidity() throws RefreshTokenException {
		IUserService userService = getUserService();
		if (userService == null) {
			throw new IllegalArgumentException("Request is not properly configured to use UserService. Use method RestConfiguration.setUserServiceClass(Class<? extends IUserService>) to deliver service implementation class");
		}
		if (!userService.isLogged()) {
			throw new IllegalArgumentException("You are not allowed to perform this action.", new RuntimeException(Request.this.getClass().getSimpleName() + " executed without logged user"));
		}
		try {
			sThreadLock.waitIfLocked();
		} catch (InterruptedException e) {
			throw new RefreshTokenException("Locked thread interrupted!", e);
		}
		if (userService.isTokenValid()) {
			return;
		}
		try {
			sThreadLock.lock();
		} catch (InterruptedException e) {
			throw new RefreshTokenException("Locked thread interrupted!", e);
		}
		try {
			userService.refreshToken();
		} catch (Exception e) {
			userService.logout();
			throw new RefreshTokenException("Problem during obtaining refresh token", e);
		} finally {
			sThreadLock.unlock();
		}
	}

	protected IUserService getUserService() {
		if (needNewUserService()) {
			Class<? extends IUserService> clazz = sConfiguration.mUserServiceClass;
			try {
				Method method = clazz.getDeclaredMethod("getInstance");
				method.setAccessible(true);
				mUserService = (IUserService) method.invoke(null, (Object[]) null);
			} catch (Exception e) {
				throw new IllegalArgumentException(clazz.getSimpleName() + " must contains public static method getInstance()");
			}
		}
		return mUserService;
	}

	private boolean needNewUserService() {
		return mUserService == null;
	}

	/**
	 * Metoda, w której powinna byc wykonana konfiguracja zapytania.
	 *
	 * @see #setUrlSegments(String...)
	 * @see #putParameter(String, Object)
	 * @see #putUrlParameter(String, Object)
	 */
	protected abstract void prepareRequest();

	/**
	 * Metoda zwracająca wynik zapytania api
	 */
	@WorkerThread
	protected abstract Response request() throws IOException;

	/**
	 * Metoda zwraca kompletny adres url do metody api. Jeśli do adresu mają zostać dodane parametry to musi to być wykonane przed
	 * wywołaniem tej funkcji.
	 *
	 * @see #putUrlParameter(String, Object)
	 */
	protected HttpUrl getUrl() {
		if (mUrlSegments == null) {
			throw new IllegalStateException("Set url segments in " + this.getClass().getSimpleName() + " in prepareRequest method");
		}

		HttpUrl.Builder builder = new HttpUrl.Builder()
				.scheme(sConfiguration.mApiScheme)
				.port(sConfiguration.mApiPort)
				.host(sConfiguration.mApiHost);
		if (!TextUtils.isEmpty(sConfiguration.mAuthorization)) {
			builder.username(sConfiguration.mAuthorizationUser)
					.password(sConfiguration.mAuthorizationPassword);
		}
		for (String segment : mUrlSegments) {
			builder.addPathSegment(segment);
		}
		for (Entry<String, Object> param : mUrlParameters) {
			builder.addQueryParameter(param.getKey(), String.valueOf(param.getValue()));
		}
		if (isTokenRequest() && getUserService() != null) {
			builder.addQueryParameter(ACCESS_TOKEN, getUserService().getToken().getToken());
		}
		return builder.build();
	}

	/**
	 * Ustawienie adresu api
	 *
	 * @param urlSegments elementy ścieżki url do odpowiedniej metody api np.: <br/>
	 *                    dla adresu http://example.com/v1/get/user metoda
	 *                    powinna być wywołana: {@code getUrlBuilder("get", "user");}
	 */
	protected void setUrlSegments(String... urlSegments) {
		mUrlSegments = urlSegments;
	}

	protected void putUrlParameter(@NonNull String parameter, @Nullable Object value) {
		mUrlParameters = new ReflectHttpUrlEncodedSerializer<>(parameter).serializeModel(value, mUrlParameters);
	}

	protected void putUrlParameter(@Nullable Object value) {
		mUrlParameters = new ReflectHttpUrlEncodedSerializer<>("").serializeModel(value, mUrlParameters);
	}

	protected void putParameter(@NonNull String parameter, @Nullable Object value) {
		mBodyParameters = new ReflectHttpUrlEncodedSerializer<>(parameter).serializeModel(value, mBodyParameters);
	}

	protected void putParameter(@Nullable Object value) {
		mBodyParameters = new ReflectHttpUrlEncodedSerializer<>("").serializeModel(value, mBodyParameters);
	}

	protected void removeParameter(String key){
		mBodyParameters.remove(key);
	}

	protected void removeUrlParameter(String key){
		mUrlParameters.remove(key);
	}

	@SuppressWarnings("SimplifiableIfStatement")
	private boolean isFileValue(Object value) {
		if (value == null) {
			return false;
		}
		if (value instanceof File) {
			return true;
		}
		if (value instanceof List && ((List) value).iterator().hasNext()
				&& ((List) value).iterator().next() instanceof File) {
			return true;
		} else {
			//noinspection ConstantConditions
			return value.getClass().isArray() && ((Object[]) value)[0] instanceof File;
		}
	}

	protected String[] toObject(String... names) {
		return names;
	}


	/**
	 * metoda zwracająca body zapytania zawierające {@code parameters}
	 *
	 * @return body zapytania lub null jeśli nie zostały przesłane żadne parametry
	 */
	@Nullable
	protected RequestBody getRequestBody() throws JsonProcessingException {
		RequestBody body;
		isMultipartRequest = isMultipartRequest();
		if (isMultipartRequest) {
			body = getMultipartBody();
		} else {
			body = getFormBody();
		}
		return body;
	}

	private boolean isMultipartRequest() {
		for (Entry<String, Object> entry : mBodyParameters) {
			if (isFileValue(entry.getValue())) {
				return true;
			}
		}
		return false;
	}

	@NonNull
	private RequestBody getMultipartBody() {
		RequestBody body;
		MultipartBody.Builder bodyBuilder = new MultipartBody.Builder();
		bodyBuilder.setType(MULTIPART_MEDIA_TYPE);

		Iterator<Entry<String, Object>> parametersIterator = mBodyParameters.iterator();
		while (parametersIterator.hasNext()) {
			Entry<String, Object> param = parametersIterator.next();
			Object value = param.getValue();
			if (isFileValue(value)) {
				if (value instanceof File) {
					addFile(bodyBuilder, param.getKey(), (File) value);
				} else if (value.getClass().isArray()) {
					addFilesList(bodyBuilder, param.getKey(), Arrays.asList((File[]) value));
				} else if (value instanceof List) {
					//noinspection unchecked
					addFilesList(bodyBuilder, param.getKey(), (List<File>) value);
				}
				parametersIterator.remove();
			}
		}

		for (Entry<String, Object> object : mBodyParameters) {
			String stringValue = obtainStringValue(object.getValue());
			Logger.debug(object.getKey() + ":", stringValue);
			if (!TextUtils.isEmpty(stringValue)) {
				bodyBuilder.addFormDataPart(object.getKey(), stringValue);
			}
		}
		return bodyBuilder.build();
	}

	protected String obtainStringValue(Object value) {
		if (value == null) {
			return null;
		}
		if (value instanceof Date) {
			return String.valueOf(((Date) value).getTime() / 1000);
		} else if (value instanceof Calendar) {
			return String.valueOf(((Calendar) value).getTimeInMillis() / 1000);
		} else if (value instanceof Boolean && sConfiguration.mCastBooleanToInt) {
			return (Boolean) value ? "1" : "0";
		} else {
			return String.valueOf(value);
		}
	}

	private void addFilesList(MultipartBody.Builder builder, String name, List<File> value) {
		for (File file : value) {
			addFile(builder, name, file);
		}
	}

	private void addFile(MultipartBody.Builder builder, String name, File file) {
		if (file != null) {
			Logger.debug(name + ":", file.getName());
			builder.addFormDataPart(name, file.getName(), RequestBody.create(MediaType.parse("image/" + getFileExtension(file)), file));
		}
	}

	private String getFileExtension(File file) {
		String fileName = file.getName();
		return fileName.substring(fileName.lastIndexOf(".") + 1);
	}

	@NonNull
	private RequestBody getFormBody() {
		RequestBody body;
		FormBody.Builder bodyBuilder = new FormBody.Builder();
		for (Entry<String, Object> object : mBodyParameters) {
			String stringValue = obtainStringValue(object.getValue());
			Logger.debug(object.getKey() + ":", stringValue);
			if (!TextUtils.isEmpty(stringValue)) {
				bodyBuilder.add(object.getKey(), stringValue);
			}
		}
		return bodyBuilder.build();
	}


}
