package com.humandevice.android.resttools.adapters;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.module.SimpleModule;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;

/**
 * Jackson Singleton.
 *
 * @author Mikołaj Styś
 * @date 2016/05/06
 */
public final class JacksonMapper {

	private final ObjectMapper mObjectMapper = new ObjectMapper();
	private final ObjectWriter mObjectWriter = mObjectMapper.writer();
	private static JacksonMapper sInstance;
	private boolean mMapperModuleRegistered = false;

	public static JacksonMapper getInstance() {
		if (sInstance == null) {
			sInstance = new JacksonMapper();
		}
		return sInstance;
	}

	private JacksonMapper() {
		mObjectMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
	}

	public void setUnixTimeDeserializer(boolean enabled){
		SimpleModule module = new SimpleModule();
		if (enabled) {
			module.addDeserializer(Calendar.class, new JsonDeserializer<Calendar>() {
				@Override
				public Calendar deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
					long value = p.getLongValue();
					Calendar calendar = Calendar.getInstance();
					calendar.setTimeInMillis(value * 1000);
					return calendar;
				}
			});
			module.addDeserializer(Date.class, new JsonDeserializer<Date>() {
				@Override
				public Date deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
					long value = p.getLongValue();
					return new Date(value * 1000);
				}
			});
		}
		setModule(module);
	}

	private void setModule(Module module){
		mMapperModuleRegistered = true;
		mObjectMapper.registerModule(module);
	}

	public ObjectReader getReaderFor(TypeReference<?> type) {
		return getObjectMapper().readerFor(type);
	}

	public ObjectReader getReaderFor(Class<?> cls) {
		return getObjectMapper().readerFor(cls);
	}

	public ObjectMapper getObjectMapper() {
		if (!mMapperModuleRegistered){
			setModule(new SimpleModule());
		}
		return mObjectMapper;
	}

	public ObjectWriter getObjectWriter() {
		return mObjectWriter;
	}
}
