package com.humandevice.android.resttools.rest;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.humandevice.android.resttools.rest.exceptions.RequestException;
import com.humandevice.android.resttools.rest.listeners.RequestListener;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Klasa opakowująca
 * {@link ThreadPoolExecutor} w celu dodania listenera do wykonywanego zadania oraz rzucania wyjątkiem {@link RequestException} w
 * przypadku otrzymania błędu z api
 *
 * @author Rafal Zajfert
 * @date 29.02.2016
 */
public class RequestExecutor extends ThreadPoolExecutor {

    public RequestExecutor() {
        this(20, 0L);
    }

    public RequestExecutor(int poolSize, long aliveDuration) {
        super(poolSize, poolSize, aliveDuration, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());
    }

    public RequestExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue);
    }

    public RequestExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory);
    }

    public RequestExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, handler);
    }

    public RequestExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
    }

    /**
     * Metoda uruchamiająca wykonywanie zadania i zwraca Future z wykonywanym zadaniem. W momencie ukończenia zadania wywoływana jest
     * metoda {@code listener}a.
     * <br />
     * Jeśli chcesz zablokować wątek na czas wykonywania zadania to należy wykonać {@code result = exec.submit(task, listener).get()}
     * @param task zadanie do wykonania
     * @param listener Listener uruchamiany w momencie ukończenia zadania
     * @param <T> typ obiektu zwracanego po prawidłowym wykonaniu zadania
     * @return Future z oczekującym zadaniem
     */
    @NonNull
    public <T> RequestFuture<T> submit(@NonNull Callable<T> task, @Nullable final RequestListener<T> listener) {
        RequestFutureTask<T> futureTask = new RequestFutureTask<>(task, listener);
        execute(futureTask);
        return futureTask;
    }

    /**
     * Metoda uruchamiająca wykonywanie zadania i zwraca Future z wykonywanym zadaniem.
     * <br />
     * Jeśli chcesz zablokować wątek na czas wykonywania zadania to należy wykonać {@code result = exec.submit(task).get()}
     * @param task zadanie do wykonania
     * @param <T> typ obiektu zwracanego po prawidłowym wykonaniu zadania
     * @return Future z oczekującym zadaniem
     */
    @NonNull
    @Override
    public <T> RequestFuture<T> submit(@NonNull Callable<T> task) {
        RequestFutureTask<T> futureTask = new RequestFutureTask<>(task);
        execute(futureTask);
        return futureTask;
    }

    /**
     * {@inheritDoc}
     * @deprecated Metoda nie jest opakowana w przechwytywanie wyjątków, należy używać metody {@link #submit(Callable, RequestListener)}
     */
    @NonNull
    @Override
    @Deprecated
    public Future<?> submit(Runnable task) {
        return super.submit(task);
    }

    /**
     * {@inheritDoc}
     * @deprecated Metoda nie jest opakowana w przechwytywanie wyjątków, należy używać metody {@link #submit(Callable, RequestListener)}
     */
    @NonNull
    @Override
    @Deprecated
    public <T> Future<T> submit(Runnable task, T result) {
        return super.submit(task, result);
    }

//    /**
//     * {@inheritDoc}
//     * @deprecated Metoda nie jest opakowana w przechwytywanie wyjątków, należy używać metody {@link #submit(Callable, RequestListener)}
//     */
//    @NonNull
//    @Override
//    @Deprecated
//    public <T> Future<T> submit(Callable<T> task) {
//        return super.submit(task);
//    }
}
