package com.humandevice.android.resttools.rest.exceptions;

import android.support.annotation.NonNull;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.rafalzajfert.androidlogger.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

/**
 * Exception rzucany w przypadku prawidłowej odpowiedzi z serwera zawierającej informacje o błędzie
 *
 * @author Rafal Zajfert
 * @date 29.02.2016
 */
@SuppressWarnings("unused")
public class RequestException extends ExecutionException {

    /**
     * Request completed
     */
    public static final int CODE_OK = 200;

    /**
     * Request accepted but data is incomplete / invalid
     */
    public static final int CODE_ACCEPTED = 202;

    /**
     * The request was invalid. You may be missing a required argument or provided bad data. An error mMessage will be returned explaining what happened.
     */
    public static final int CODE_BAD_REQUEST = 400;
    /**
     * The authentication you provided is invalid.
     */
    public static final int CODE_UNAUTHORIZED = 401;
    /**
     * You don't have permission to complete the operation or access the resource.
     */
    public static final int CODE_FORBIDDEN = 403;
    /**
     * You requested an invalid method.
     */
    public static final int CODE_NOT_FOUND = 404;
    /**
     * The method specified in the Request-Line is not allowed for the resource identified by the Request-URI. (used POST instead of PUT)
     */
    public static final int CODE_METHOD_NOT_ALLOWED = 405;
    /**
     * The server timed out waiting for the request.
     */
    public static final int CODE_TIMEOUT = 408;
    /**
     * You have exceeded the rate limit.
     */
    public static final int CODE_TOO_MANY_REQUESTS = 429;
    /**
     * Something is wrong on our end. We'll investigate what happened.
     */
    public static final int CODE_INTERNAL_SERVER_ERROR = 500;
    /**
     * The method you requested is currently unavailable (due to maintenance or high load).
     */
    public static final int CODE_SERVICE_UNAVAILABLE = 503;
    /**
     * Non server exception
     */
    public static final int UNKNOWN = -1;
    /**
     * Waiting thread is activated before the condition it was waiting for has been satisfied.
     */
    public static final int INTERRUPTED = -2;
    /**
     * Blocking operation times out
     */
    public static final int TIMEOUT = -3;

    @NonNull
    protected ErrorResponse mErrorResponse;

    public RequestException(@NonNull Exception e) {
        super(e);
        mErrorResponse = new ErrorResponse();
        mErrorResponse.setName("Unknown Error");
        mErrorResponse.setMessage(e.getMessage());
        mErrorResponse.setCode(UNKNOWN);
        if (e instanceof InterruptedException) {
            mErrorResponse.setStatusCode(INTERRUPTED);
        } else if (e instanceof TimeoutException) {
            mErrorResponse.setStatusCode(TIMEOUT);
        } else {
            mErrorResponse.setStatusCode(UNKNOWN);
        }
        mErrorResponse.setErrors(new HashMap<String, String[]>());
    }

    /**
     * Konstruktor wyjątku parsujący odpowiedź z serwera
     */
    public RequestException(@NonNull String errorJson, int requestCode) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        ObjectReader errorReader = objectMapper.readerFor(ErrorResponse.class);
        mErrorResponse = errorReader.readValue(errorJson);
        mErrorResponse.setStatusCode(requestCode);
    }

    /**
     * Metoda sprawdzająca czy exception został wywołany na etapie połączenia z serwerem
     */
    public boolean isConnectionError() {
        return mErrorResponse.mStatusCode <= 0;
    }

    /**
     * Metoda sprawdzająca czy exception został wywołany na serwerze i został zwrócony json błędu
     */
    public boolean isResponseError() {
        return mErrorResponse.mStatusCode > 0;
    }

    /**
     * Wiadomość błędu
     */
    @Override
    public String getMessage() {
        return mErrorResponse.mMessage;
    }

    /**
     * Kod błędu
     */
    public int getCode() {
        return mErrorResponse.mCode;
    }

    /**
     * Kod odpowiedzi
     */
    public int getStatusCode() {
        return mErrorResponse.mStatusCode;
    }

    /**
     * błędy w formularzu
     *
     * @return mapa błędów gdzie klucz to nazwa pola, a wartość to błędy
     */
    public Map<String, String[]> getErrors() {
        return mErrorResponse.mErrors;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder(getMessage());
        for (Map.Entry<String, String[]> entry : getErrors().entrySet()) {
            builder.append("\n");
            builder.append(entry.getKey()).append(": ").append(Arrays.toString(entry.getValue()));
        }
        return builder.toString();
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ErrorResponse {

        /**
         * Nazwa błędu
         */
        @JsonProperty("name")
        private String mName;

        /**
         * Wiadomość dla użytkownika
         */
        @JsonProperty("message")
        private String mMessage;

        /**
         * Kod błędu
         */
        @JsonProperty("code")
        private int mCode;

        /**
         * Kod odpowiedzi
         */
        @JsonProperty("status")
        private int mStatusCode;

        @JsonProperty("errors")
        private Map<String, String[]> mErrors = new HashMap<>();

        public void setMessage(String message) {
            mMessage = message;
        }

        public void setCode(int code) {
            mCode = code;
        }

        public void setStatusCode(int statusCode) {
            mStatusCode = statusCode;
        }

        public void setErrors(Map<String, String[]> errors) {
            this.mErrors = errors;
        }

        @JsonSetter("errors")
        public void parseErrors(Map<String, List<Object>> errors) {
            Logger.error("parse errors: " + errors);
            if (errors == null) {
                return;
            }
            mErrors.putAll(parseErrorsMap(null, errors));

        }

        private Map<String, String[]> parseErrorsMap(String objectName, Map<String, List<Object>> errors){
            Map<String, String[]> resultMap = new HashMap<>();

            for (Map.Entry<String, List<Object>> entry : errors.entrySet()) {
                if (entry.getValue() != null && !entry.getValue().isEmpty()) {
                    if (entry.getValue().get(0) instanceof String) {
                        String[] values = new String[entry.getValue().size()];
                        for (int i = 0; i < entry.getValue().size(); i++) {
                            values[i] = String.valueOf(entry.getValue().get(0));
                        }
                        String key = entry.getKey();
                        if (objectName != null){
                            key = objectName + "[" + key + "]";
                        }

                        resultMap.put(key, values);
                    } else if (entry.getValue().get(0) instanceof Map) {
                        String key = entry.getKey();
                        if (objectName != null){
                            key = objectName + "[" + key + "]";
                        }
                        for (Object map : entry.getValue()) {
                            resultMap.putAll(parseErrorsMap(key, (Map<String, List<Object>>) map));
                        }
                    }
                }
            }
            return resultMap;
        }

        public void setName(String name) {
            mName = name;
        }
    }
}
