package com.humandevice.android.steepprogressview.views;

import android.animation.Animator;
import android.animation.ValueAnimator;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.AccelerateDecelerateInterpolator;

import com.humandevice.android.steepprogressview.R;
import com.humandevice.android.steepprogressview.views.listeners.SimpleAnimatorListener;
import com.rafalzajfert.androidlogger.Logger;

/**
 * View that provides circle steps connected with lines. Helpful in forms where user want to know how many steps left to go.
 * There are several attributes to customize layout.
 *
 * spv_neutralColor - color of neutral (not yet accessed) steps and all lines between them.
 * spv_neutralBorderColor - color of neutral (not yet accessed) steps border.
 * spv_activeColor - color of active (currently shown) step.
 * spv_activeBorderColor - color of active (currently shown) step border.
 * spv_pastColor - color of past (already done) steps and line between current and last step.
 * spv_pastBorderColor - color of past (already done) steps border.
 * spv_textColor - color of all numbers in circles (there is no emphasis between past, current and neutral steps).
 * spv_textSize - size of all numbers in circles. If flag spv_noText is set to true and spv_circleRadius is not set, this regulates circle
 * size.
 * spv_stepSize - number of steps (how many steps will be drawn on layout). Default value is 3.
 * spv_currentStep - current active step. This should not be bigger than stepSize. Default value is 1.
 * spv_lineWidth - width of line stroke. This defines how thick line between steps should be.
 * spv_borderSize - width of border around steps. If set to 0, there will be no border. Default value is 0.
 * spv_noText - if set to true no text will be drawn. Default value is false. If you want to set smaller circle size change spv_textSize
 * attribute.
 * spv_circleRadius - size of the background circle. If not set, circle will be adjusted to text size.
 *
 * @author Szymon Bartczak
 * @date 2016-08-17
 */
public class StepProgressView extends View {

    private static final int DEFAULT_NEUTRAL_CIRCLE_COLOR = Color.BLUE;
    private static final int DEFAULT_ACTIVE_CIRCLE_COLOR = Color.YELLOW;
    private static final int DEFAULT_PAST_CIRCLE_COLOR = Color.GREEN;
    private static final int DEFAULT_NEUTRAL_BORDER_COLOR = Color.parseColor("#90CAF9");
    private static final int DEFAULT_ACTIVE_BORDER_COLOR = Color.parseColor("#FFF59D");
    private static final int DEFAULT_PAST_BORDER_COLOR = Color.parseColor("#A5D6A7");
    private static final int DEFAULT_TEXT_COLOR = Color.WHITE;
    private static final int DEFAULT_STEP_SIZE = 3;
    private static final int MINIMUM_STEP_SIZE = 2;
    private static final float DEFAULT_TEXT_SIZE = 40;//in px??
    private static final int MINIMUM_LINES_SIZE = 50;
    private static final int DEFAULT_LINE_WIDTH = 2;
    private static final boolean DEFAULT_NO_TEXT = false;
    private static final int NO_CIRCLE_RADIUS = -1;
    private static final int DEFAULT_BORDER_SIZE = 0;

    private Context mContext;

    //paints
    private Paint mNeutralPaint;
    private Paint mActivePaint;
    private Paint mPastPaint;
    private Paint mTextPaint;
	private Paint mNeutralBorderPaint;
    private Paint mActiveBorderPaint;
    private Paint mPastBorderPaint;
    //---------------------

    //attrs
    private int mNeutralCircleColor;
    private int mActiveCircleColor;
    private int mPastCircleColor;
    private int mNeutralBorderColor;
    private int mActiveBorderColor;
    private int mPastBorderColor;
    private int mTextColor;
    private int mStepSize;
    private float mTextSize;
    private int mLineWidth;
    private boolean mNoText;
    //---------------------

    //draw measurement
    private float mCircleRadius;
    private float mSingleLineLength;//length of single line between two steps
    private float mLineLength;//length of the line from center of first step to center of last step
    private int mBorderSize;
    //---------------------

    private int mCurrentStep;//counted as [1,mStepSize]
    private float mY;
    private float mX;
    private float mProgressLength;
    private boolean mAnimationStarted = false;
    private ValueAnimator mAnimator;


    public StepProgressView(Context context) {
        //super(context);
        this(context, null);
    }

    public StepProgressView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public StepProgressView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mContext = context;
        initAttrs(context, attrs);
        initPaint();
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    public StepProgressView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        mContext = context;
        initAttrs(context, attrs);
        initPaint();
    }

    private void initPaint() {
        mNeutralPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mNeutralPaint.setColor(mNeutralCircleColor);
        mNeutralPaint.setStyle(Paint.Style.FILL);
        mNeutralPaint.setStrokeWidth(mLineWidth);

        mActivePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mActivePaint.setColor(mActiveCircleColor);
        mActivePaint.setStyle(Paint.Style.FILL);

        mPastPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPastPaint.setColor(mPastCircleColor);
        mPastPaint.setStyle(Paint.Style.FILL);
        mPastPaint.setStrokeWidth(mLineWidth);

		mNeutralBorderPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mNeutralBorderPaint.setColor(mNeutralBorderColor);
        mNeutralBorderPaint.setStyle(Paint.Style.STROKE);
        mNeutralBorderPaint.setStrokeWidth(mBorderSize);

        mActiveBorderPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mActiveBorderPaint.setColor(mActiveBorderColor);
        mActiveBorderPaint.setStyle(Paint.Style.STROKE);
        mActiveBorderPaint.setStrokeWidth(mBorderSize);

        mPastBorderPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPastBorderPaint.setColor(mPastBorderColor);
        mPastBorderPaint.setStyle(Paint.Style.STROKE);
        mPastBorderPaint.setStrokeWidth(mBorderSize);

        mTextPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mTextPaint.setTextSize(mTextSize);
        mTextPaint.setColor(mTextColor);
        mTextPaint.setTextAlign(Paint.Align.CENTER);
    }

    private void initAttrs(Context context, AttributeSet attrs) {
        TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.StepProgressView, 0, 0);

        mNeutralCircleColor = typedArray.getColor(R.styleable.StepProgressView_spv_neutralColor, DEFAULT_NEUTRAL_CIRCLE_COLOR);
        mActiveCircleColor = typedArray.getColor(R.styleable.StepProgressView_spv_activeColor, DEFAULT_ACTIVE_CIRCLE_COLOR);
        mPastCircleColor = typedArray.getColor(R.styleable.StepProgressView_spv_pastColor, DEFAULT_PAST_CIRCLE_COLOR);
        mNeutralBorderColor = typedArray.getColor(R.styleable.StepProgressView_spv_neutralBorderColor, DEFAULT_NEUTRAL_BORDER_COLOR);
        mActiveBorderColor = typedArray.getColor(R.styleable.StepProgressView_spv_activeBorderColor, DEFAULT_ACTIVE_BORDER_COLOR);
        mPastBorderColor = typedArray.getColor(R.styleable.StepProgressView_spv_pastBorderColor, DEFAULT_PAST_BORDER_COLOR);
        mTextColor = typedArray.getColor(R.styleable.StepProgressView_spv_textColor, DEFAULT_TEXT_COLOR);
        mTextSize = typedArray.getDimension(R.styleable.StepProgressView_spv_textSize, DEFAULT_TEXT_SIZE);
        mCircleRadius = typedArray.getDimensionPixelSize(R.styleable.StepProgressView_spv_circleRadius, NO_CIRCLE_RADIUS);
        mBorderSize = typedArray.getDimensionPixelSize(R.styleable.StepProgressView_spv_borderSize, DEFAULT_BORDER_SIZE);
        mStepSize = typedArray.getInt(R.styleable.StepProgressView_spv_stepSize, DEFAULT_STEP_SIZE);
        mCurrentStep = typedArray.getInt(R.styleable.StepProgressView_spv_currentStep, 1);
        mNoText = typedArray.getBoolean(R.styleable.StepProgressView_spv_noText, DEFAULT_NO_TEXT);
        mLineWidth = typedArray.getDimensionPixelSize(R.styleable.StepProgressView_spv_lineWidth, DEFAULT_LINE_WIDTH);
        if (mStepSize < MINIMUM_STEP_SIZE) {
            throw new IllegalStateException("Step size is smaller than minimum: " + MINIMUM_STEP_SIZE + ".");
        }
        if (mCurrentStep > mStepSize) {
            throw new IllegalStateException("Current step is greater than step size.");
        }

        if(mCircleRadius == NO_CIRCLE_RADIUS) {
            mCircleRadius = (mContext.getResources().getDisplayMetrics().scaledDensity * mTextSize) / 2;
        }

        typedArray.recycle();
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
		w = w-getPaddingLeft()-getPaddingRight();
        mLineLength = w - 2 * mCircleRadius;
        mSingleLineLength = (w - 2 * mCircleRadius * mStepSize) / (mStepSize - 1);
        mX = getPaddingLeft() + mCircleRadius;
        mY = getPaddingTop() + mCircleRadius;
        mProgressLength = (mSingleLineLength + 2 * mCircleRadius)*(mCurrentStep-1);
    }

    @Override
    protected int getSuggestedMinimumWidth() {
        return (int) (2 * mCircleRadius * mStepSize + MINIMUM_LINES_SIZE * (mStepSize - 1));
    }

    @Override
    protected int getSuggestedMinimumHeight() {
        return (int) (2 * mCircleRadius);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int w;
        int h;

        int minw = getPaddingLeft() + getPaddingRight() + getSuggestedMinimumWidth();
        int minh = getPaddingTop() + getPaddingBottom() + getSuggestedMinimumHeight();

        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int widthSize = MeasureSpec.getSize(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int heightSize = MeasureSpec.getSize(heightMeasureSpec);

        if (widthMode == MeasureSpec.EXACTLY) {
            w = widthSize;
        } else if (widthMode == MeasureSpec.AT_MOST) {
            w = Math.min(minw, widthSize);
        } else {
            w = minw;
        }

        if (heightMode == MeasureSpec.EXACTLY) {
            h = heightSize;
        } else {
            h = minh;
        }

        setMeasuredDimension(w, h);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        //drawing lines
        canvas.drawLine(mX, mY, mX + mLineLength, mY, mNeutralPaint);//neutral line
        canvas.drawLine(mX, mY, mX + mProgressLength, mY, mPastPaint);//past line
        //canvas.drawLine(x, y, x + mLineLength, y, mBorderPaint);//border todo attrs to change border

        //drawing circles and text
        for (int i = 1; i <= mStepSize; i++) {
            Paint currentPaint;
            Paint currentBorder;
            float circlex = mX + (mSingleLineLength + 2 * mCircleRadius) * (i - 1);
            float progressLengthToThisPoint = (mSingleLineLength + 2 * mCircleRadius)*(i-1);

            if(mProgressLength < progressLengthToThisPoint) {
                currentPaint = mNeutralPaint;
                currentBorder = mNeutralBorderPaint;
            } else if (mProgressLength == progressLengthToThisPoint) {
                currentPaint = mActivePaint;
                currentBorder = mActiveBorderPaint;
            } else {
                currentPaint = mPastPaint;
                currentBorder = mPastBorderPaint;
            }

            canvas.drawCircle(circlex, mY, mCircleRadius, currentPaint);
            if (mBorderSize > 0) {
                canvas.drawCircle(circlex, mY, mCircleRadius, currentBorder);
            }

            if(!mNoText) {
                canvas.drawText(String.valueOf(i), circlex, mY + mTextSize / 3, mTextPaint);
            }
        }
    }

    public void next(){
        setStep(mCurrentStep+1);
    }

    public void previous(){
        setStep(mCurrentStep-1);
    }

    public void setStep(int step) {
        if (step > mStepSize || step <= 0) {
            Logger.error(new IllegalStateException("Current step is smaller than 1 or greater than step size."));
            return;
        }
		mCurrentStep = step;
		if(mLineLength == 0) { //prevents drawing when view has not been properly measured yet. It wa
			return;
		}
		float lastProgressLength = mProgressLength;
        mProgressLength = (mSingleLineLength + 2 * mCircleRadius)*(mCurrentStep-1);
        if(mAnimationStarted) {
            cancelCurrentAnimation();
        }
        animateStepTransition(lastProgressLength, mProgressLength);
    }

    public int getCurrentStep() {
        return mCurrentStep;
    }

    private void cancelCurrentAnimation() {
        mAnimator.cancel();
    }

    private void animateStepTransition(float from, float to) {
        mAnimator = ValueAnimator.ofFloat(from, to).setDuration(300);
        mAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                float value = (Float) (valueAnimator.getAnimatedValue());
                mProgressLength = value;
                invalidate();
            }
        });
        mAnimator.addListener(new SimpleAnimatorListener() {
            @Override
            public void onAnimationStart(Animator animator) {
                mAnimationStarted = true;
            }

            @Override
            public void onAnimationEnd(Animator animator) {
                mAnimationStarted = false;
            }

            @Override
            public void onAnimationCancel(Animator animator) {
                Logger.trace();
            }
        });
        mAnimator.setInterpolator(new AccelerateDecelerateInterpolator());
        mAnimator.start();
    }
}
