package com.humandevice.android.stringtools.stringtools;

import android.graphics.Paint;
import android.graphics.Typeface;
import android.os.Parcel;
import android.text.TextPaint;
import android.text.style.TypefaceSpan;

import java.util.WeakHashMap;

/**
 * Class for building custom font Strings.
 *
 * @author Rafal Zajfert
 */
public class SpannableFont extends StyledSpannableString {

	//region Constructors
	public SpannableFont() {
		super();
	}

	public SpannableFont(CharSequence text) {
		super(text);
	}

	public SpannableFont(CharSequence text, String fontAsset) {
		appendStyled(text, new FontSpan(fontAsset));
	}
	//endregion

	//region Append methods
	public SpannableFont appendFont(CharSequence text, String fontAsset) {
		appendStyled(text, new FontSpan(fontAsset));
		return this;
	}
	//endregion

	private static class FontSpan extends TypefaceSpan {

		private static final WeakHashMap<String, Typeface> mFontCache = new WeakHashMap<>();
		private String mFontAsset;

		public FontSpan(String fontAsset) {
			super("custom");
			mFontAsset = fontAsset;
		}
		public FontSpan(Parcel src) {
			super(src);
			mFontAsset = src.readString();
		}

		private Typeface getTypeface() {
			Typeface typeface;
			if (mFontCache.containsKey(mFontAsset)) {
				typeface = mFontCache.get(mFontAsset);
			} else {
				typeface = Typeface.createFromAsset(getContext().getAssets(), mFontAsset);
				mFontCache.put(mFontAsset, typeface);
			}
			return typeface;
		}


		public void writeToParcel(Parcel dest, int flags) {
			super.writeToParcel(dest, flags);
			dest.writeString(mFontAsset);
		}

		@Override
		public void updateDrawState(final TextPaint drawState) {
			apply(drawState);
		}

		@Override
		public void updateMeasureState(final TextPaint paint) {
			apply(paint);
		}

		private void apply(final Paint paint) {
			final Typeface oldTypeface = paint.getTypeface();
			final Typeface typeface = getTypeface();
			final int oldStyle = oldTypeface != null ? oldTypeface.getStyle() : 0;
			final int fakeStyle = oldStyle & ~typeface.getStyle();

			if ((fakeStyle & Typeface.BOLD) != 0) {
				paint.setFakeBoldText(true);
			}

			if ((fakeStyle & Typeface.ITALIC) != 0) {
				paint.setTextSkewX(-0.25f);
			}

			paint.setTypeface(typeface);
		}
	}
}
