package com.humandevice.validator.rules;

import android.app.Application;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;

import com.humandevice.validator.errors.ValidationException;
import com.humandevice.validator.widget.ValidatorView;

import java.util.Comparator;

/**
 * Abstract rule implemented by concrete Rules.
 * @author  Rafal Zajfert
 */
public abstract class Rule<T extends Rule<T>> {

    @Nullable
    private static Context context;

    @NonNull
    protected String message = "Field contains errors";

    private int priority;

    public Rule(){
    }

    public final void validate(ValidatorView view) throws ValidationException {
        checkInstance(view);
        checkView(view);
    }

    protected abstract void checkView(ValidatorView view) throws ValidationException;

    protected abstract void checkInstance(ValidatorView view) throws IllegalArgumentException;

    @NonNull
    public T setErrorMessage(@NonNull String message) {
        this.message = message;
        //noinspection unchecked
        return (T)this;
    }

    @NonNull
    public T setErrorMessage(@NonNull String message, Object... formatArgs) {
        this.message = String.format(message, formatArgs);
        //noinspection unchecked
        return (T)this;
    }

    @NonNull
    public T setErrorMessage(@StringRes int messageRes) {
        this.message = getContext().getResources().getString(messageRes);
        //noinspection unchecked
        return (T)this;
    }

    @NonNull
    public T setErrorMessage(@StringRes int messageRes, Object... formatArgs) {
        this.message = getContext().getResources().getString(messageRes, formatArgs);
        //noinspection unchecked
        return (T)this;
    }

    @NonNull
    public T setPriority(int priority){
        this.priority = priority;
        //noinspection unchecked
        return (T)this;
    }

    public int getPriority() {
        return priority;
    }

    @NonNull
    private static Context getContext() {
        if (context == null) {
            try {
                context = (Application) Class.forName("android.app.ActivityThread").getMethod("currentApplication").invoke(null, (Object[]) null);
            } catch (Exception e) {
                throw new IllegalArgumentException("Cannot find Application context. Please call Rule.setConext(Context) before");
            }
        }
        return context;
    }

    public static void setContext(@NonNull Context context) {
        Rule.context = context;
    }

    public static class PriorityComparator implements Comparator<Rule> {

        @Override
        public int compare(Rule lhs, Rule rhs) {
            if (lhs.priority != rhs.priority) {
                return compareInt(lhs.priority, rhs.priority);
            } else {
                return compareInt(lhs.hashCode(), rhs.hashCode());
            }
        }

        public static int compareInt(int lhs, int rhs) {
            return lhs < rhs ? -1 : (lhs == rhs ? 0 : 1);
        }
    }
}
