package com.humandevice.validator;

import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.res.Resources;
import android.os.Build;
import android.support.annotation.IntDef;
import android.support.design.widget.Snackbar;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.KeyCharacterMap;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.Toast;

import com.humandevice.validator.errors.Errors;
import com.humandevice.validator.widget.ValidatorView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Set;

/**
 * Created by Rafal Zajfert on 10.08.2015.
 */
@SuppressWarnings("unused")
public class Validator {
	
	public static final int PRIORITY_ALL = Integer.MAX_VALUE;
	
	public static final int TOAST = 0;
	public static final int SNACK = 1;
	public static final int LENGTH_SHORT = 0;
	public static final int LENGTH_LONG = 1;
	public static final int LENGTH_INDEFINITE = 2;
	
	public static final String TAG = Validator.class.getSimpleName();
	
	@IntDef({Validator.TOAST, Validator.SNACK})
	@Retention(RetentionPolicy.SOURCE)
	public @interface DisplayMode {
	}
	
	@IntDef({LENGTH_SHORT, LENGTH_LONG, LENGTH_INDEFINITE})
	@Retention(RetentionPolicy.SOURCE)
	public @interface Duration {
	}
	
	private Validator() {
	}
	
	/**
	 * @return true jeśli wszystkie pola są uzupełnione prawidłowo
	 */
	public static boolean validate(View rootView) {
		return validate(rootView, PRIORITY_ALL);
	}
	
	public static boolean validate(View rootView, int maxPriority) {
		boolean valid = true;
		if (rootView instanceof ValidatorView) {
			valid = ((ValidatorView) rootView).validate(maxPriority);
		} else if (rootView instanceof ViewGroup) {
			for (int i = 0; i < ((ViewGroup) rootView).getChildCount(); i++) {
				boolean validView = validate(((ViewGroup) rootView).getChildAt(i), maxPriority);
				if (valid) {
					valid = validView;
				}
			}
		}
		return valid;
	}
	
	public static void showErrors(View rootView, Errors errors) {
		showErrors(rootView, errors, TOAST, LENGTH_LONG);
	}
	
	public static void showErrors(View rootView, Errors errors, @DisplayMode int otherErrorDisplayMode) {
		showErrors(rootView, errors, otherErrorDisplayMode, LENGTH_LONG);
	}
	
	public static void showErrors(View rootView, Errors errors, @DisplayMode int otherErrorDisplayMode, @Duration int otherErrorDisplayDuration) {
		Set<String> fieldNames = errors.getFieldNamesWithErrors();
		
		View focusView = showChildErrors(rootView, errors, null, fieldNames);
		for (String name : fieldNames) {
			showGeneralError(rootView, errors.getCombinedMessageWithName(name), otherErrorDisplayMode, otherErrorDisplayDuration);
		}
		
		if (focusView != null) {
			focusView.requestFocus();
		}
	}
	
	private static View showChildErrors(View rootView, Errors errors, View focusView, Set<String> fieldNames) {
		if (rootView instanceof ValidatorView) {
			((ValidatorView) rootView).getErrors().clear();
			((ValidatorView) rootView).getErrors().addAll(errors.getErrorsForName(((ValidatorView) rootView).getName()));
			
			if (((ValidatorView) rootView).hasErrors()) {
				focusView = rootView;
			}
			showViewErrors((ValidatorView) rootView);
			fieldNames.remove(((ValidatorView) rootView).getName());
		} else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				focusView = showChildErrors(view, errors, focusView, fieldNames);
			}
		}
		return focusView;
	}
	
	public static void showErrors(View rootView) {
		View focusView = showChildErrors(rootView, null);
		if (focusView != null) {
			focusView.requestFocus();
		}
	}
	
	private static View showChildErrors(View rootView, View focusView) {
		if (rootView instanceof ValidatorView) {
			if (((ValidatorView) rootView).hasErrors()) {
				focusView = rootView;
			}
			showViewErrors((ValidatorView) rootView);
		} else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				focusView = showChildErrors(view, focusView);
			}
		}
		return focusView;
	}
	
	public static boolean hasErrors(View rootView) {
		return hasChildErrors(rootView, false);
	}
	
	private static boolean hasChildErrors(View rootView, boolean hasErrors) {
		if (rootView instanceof ValidatorView) {
			hasErrors = ((ValidatorView) rootView).hasErrors();
		} else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				if (hasChildErrors(view, hasErrors)) {
					hasErrors = true;
					break;
				}
			}
		}
		return hasErrors;
	}
	
	private static void showViewErrors(ValidatorView view) {
		view.showErrors();
	}
	
	private static void showGeneralError(View parent, String message, @DisplayMode int otherErrorDisplayMode, @Duration int otherErrorDisplayDuration) {
		if (otherErrorDisplayMode == SNACK) {
			//noinspection WrongConstant
			showSnackbar(parent, message, getSnackDuration(otherErrorDisplayDuration));
		} else if (otherErrorDisplayMode == TOAST) {
			//noinspection WrongConstant
			Toast.makeText(parent.getContext(), message, getToastDuration(otherErrorDisplayDuration)).show();
		}
	}
	
	private static void showSnackbar(View parent, String message, @Duration int duration) {
		Snackbar snackbar = Snackbar.make(parent, message, duration);
		Activity activity = getActivity(parent.getContext());
		if (activity != null && isTranslucentNavigationBar(activity)) {
			final FrameLayout snackBarView = (FrameLayout) snackbar.getView();
			final FrameLayout.LayoutParams params = (FrameLayout.LayoutParams) snackBarView.getChildAt(0).getLayoutParams();
			params.setMargins(params.leftMargin,
					params.topMargin,
					params.rightMargin,
					params.bottomMargin + getNavigationBarHeight(parent));
			
			snackBarView.getChildAt(0).setLayoutParams(params);
		}
		snackbar.show();
	}
	
	private static boolean isTranslucentNavigationBar(Activity activity) {
		Window w = activity.getWindow();
		WindowManager.LayoutParams lp = w.getAttributes();
		int flags = lp.flags;
		return !hasNavBar(activity) && android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.KITKAT
				&& (flags & WindowManager.LayoutParams.FLAG_TRANSLUCENT_NAVIGATION) == WindowManager.LayoutParams.FLAG_TRANSLUCENT_NAVIGATION;
	}
	
	private static int getNavigationBarHeight(View parent) {
		Resources resources = parent.getContext().getResources();
		int resourceId = resources.getIdentifier("navigation_bar_height", "dimen", "android");
		
		if (resourceId > 0) {
			return resources.getDimensionPixelSize(resourceId);
		}
		return 0;
	}
	
	private static boolean hasNavBar(Activity activity) {
		boolean hasSoftwareKeys;
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
			Display d = activity.getWindowManager().getDefaultDisplay();
			DisplayMetrics realDisplayMetrics = new DisplayMetrics();
			d.getRealMetrics(realDisplayMetrics);
			int realHeight = realDisplayMetrics.heightPixels;
			int realWidth = realDisplayMetrics.widthPixels;
			DisplayMetrics displayMetrics = new DisplayMetrics();
			d.getMetrics(displayMetrics);
			int displayHeight = displayMetrics.heightPixels;
			int displayWidth = displayMetrics.widthPixels;
			hasSoftwareKeys = (realWidth - displayWidth) > 0 || (realHeight - displayHeight) > 0;
		} else {
			boolean hasMenuKey = ViewConfiguration.get(activity).hasPermanentMenuKey();
			boolean hasBackKey = KeyCharacterMap.deviceHasKey(KeyEvent.KEYCODE_BACK);
			hasSoftwareKeys = !hasMenuKey && !hasBackKey;
		}
		return hasSoftwareKeys;
	}
	
	private static Activity getActivity(Context context) {
		while (context instanceof ContextWrapper) {
			if (context instanceof Activity) {
				return (Activity) context;
			}
			context = ((ContextWrapper) context).getBaseContext();
		}
		return null;
	}
	
	
	private static int getSnackDuration(@Duration int otherErrorDisplayDuration) {
		switch (otherErrorDisplayDuration) {
			case LENGTH_SHORT:
				return Snackbar.LENGTH_SHORT;
			case LENGTH_INDEFINITE:
				return Snackbar.LENGTH_INDEFINITE;
			case LENGTH_LONG:
				return Snackbar.LENGTH_LONG;
			default:
				return otherErrorDisplayDuration;
		}
	}
	
	private static int getToastDuration(@Duration int otherErrorDisplayDuration) {
		switch (otherErrorDisplayDuration) {
			case LENGTH_SHORT:
				return Toast.LENGTH_SHORT;
			case LENGTH_INDEFINITE:
			case LENGTH_LONG:
			default:
				return Toast.LENGTH_LONG;
		}
	}
}
