package com.humandevice.validator;

import android.support.annotation.IntDef;
import android.support.design.widget.Snackbar;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.humandevice.validator.errors.Errors;
import com.humandevice.validator.widget.ValidatorView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Set;

/**
 * Created by Rafal Zajfert on 10.08.2015.
 */
@SuppressWarnings("unused")
public class Validator {

	public static final int PRIORITY_ALL = Integer.MAX_VALUE;

	public static final int TOAST = 0;
	public static final int SNACK = 1;
	public static final String TAG = Validator.class.getSimpleName();

	@IntDef({Validator.TOAST, Validator.SNACK})
	@Retention(RetentionPolicy.SOURCE)
	public @interface DisplayMode {
	}

	private Validator() {
	}

	/**
	 * @return true jeśli wszystkie pola są uzupełnione prawidłowo
	 */
	public static boolean validate(View rootView) {
		return validate(rootView, PRIORITY_ALL);
	}

	public static boolean validate(View rootView, int maxPriority) {
		boolean valid = true;
		if (rootView instanceof ValidatorView) {
			valid = ((ValidatorView) rootView).validate(maxPriority);
		}else if (rootView instanceof ViewGroup) {
			for (int i = 0; i < ((ViewGroup) rootView).getChildCount(); i++) {
				boolean validView = validate(((ViewGroup) rootView).getChildAt(i), maxPriority);
				if (valid) {
					valid = validView;
				}
			}
		}
		return valid;
	}

	public static void showErrors(View rootView, Errors errors) {
		showErrors(rootView, errors, TOAST);
	}

	public static void showErrors(View rootView, Errors errors, @DisplayMode int generalErrorDisplayMode) {
		Set<String> fieldNames = errors.getFieldNamesWithErrors();

		View focusView = showChildErrors(rootView, errors, null, fieldNames);
		for (String name : fieldNames) {
			showGeneralError(rootView, errors.getCombinedMessageWithName(name), generalErrorDisplayMode);
		}

		if (focusView != null) {
			focusView.requestFocus();
		}
	}

	private static View showChildErrors(View rootView, Errors errors, View focusView, Set<String> fieldNames) {
		Log.d(TAG, "showChildErrors(" + rootView  + ")");
		if (rootView instanceof ValidatorView) {
			Log.d(TAG, "showChildErrors(" + ((ValidatorView) rootView).getName()  + ")");
			((ValidatorView) rootView).getErrors().clear();
			((ValidatorView) rootView).getErrors().addAll(errors.getErrorsForName(((ValidatorView) rootView).getName()));

			if (((ValidatorView) rootView).hasErrors()) {
				focusView = rootView;
			}
			showViewErrors((ValidatorView) rootView);
			fieldNames.remove(((ValidatorView) rootView).getName());
		} else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				focusView = showChildErrors(view, errors, focusView, fieldNames);
			}
		}
		return focusView;
	}

	public static void showErrors(View rootView) {
		Log.d(TAG, "showErrors(" + rootView + ")");
		View focusView = showChildErrors(rootView, null);
		if (focusView != null) {
			focusView.requestFocus();
		}
	}

	private static View showChildErrors(View rootView, View focusView) {
		Log.d(TAG, "showChildErrors(" + rootView + ", " +focusView + ")");
		if (rootView instanceof ValidatorView){
			Log.d(TAG, "showChildErrors(" + ((ValidatorView) rootView).getName() + ")");
			if (((ValidatorView) rootView).hasErrors()) {
				focusView = rootView;
			}
			showViewErrors((ValidatorView) rootView);
		}else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				focusView = showChildErrors(view, focusView);
			}
		}
		return focusView;
	}

	public static boolean hasErrors(View rootView) {
		return hasChildErrors(rootView, false);
	}

	private static boolean hasChildErrors(View rootView, boolean hasErrors) {
		Log.d(TAG, "hasChildErrors(" + rootView + ", " +hasErrors + ")");
		if (rootView instanceof ValidatorView) {
			Log.d(TAG, "hasChildErrors(" + ((ValidatorView) rootView).getName() + ")");
			hasErrors = ((ValidatorView) rootView).hasErrors();
		} else if (rootView instanceof ViewGroup) {
			for (int i = ((ViewGroup) rootView).getChildCount() - 1; i >= 0; i--) {
				View view = ((ViewGroup) rootView).getChildAt(i);
				if (hasChildErrors(view, hasErrors)) {
					hasErrors = true;
					break;
				}
			}
		}
		return hasErrors;
	}

	private static void showViewErrors(ValidatorView view) {
		Log.d(TAG, "showViewErrors(" + view.getName() + ")");
		view.showErrors();
	}

	private static void showGeneralError(View parent, String message, @DisplayMode int generalErrorDisplayMode) {
		if (generalErrorDisplayMode == SNACK) {
			Snackbar.make(parent, message, Snackbar.LENGTH_SHORT).show();
		} else if (generalErrorDisplayMode == TOAST) {
			Toast.makeText(parent.getContext(), message, Toast.LENGTH_LONG).show();
		}
	}
}
