package com.mwstys.android.permission;

import android.app.Activity;
import android.app.Fragment;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.v13.app.FragmentCompat;
import android.support.v4.app.ActivityCompat;

import com.humandevice.android.core.view.ToolsFragment;
import com.humandevice.android.core.view.interfaces.ContextOperator;


/**
 * Permission system wrapper
 * Date: 2016/02/18
 *
 * @author Mikołaj Styś
 */
public class PermissionHelper implements PermissionsResultListener {

    private static final int PERMISSIONS_REQUEST_CODE = 0xAA;
    private final ContextOperator mContextOperator;
    private OnPermissionGranted mListener;
    private RationaleListener mRationaleListener;

    public PermissionHelper(@NonNull ContextOperator context) {
        this.mContextOperator = context;
    }


    public PermissionHelper setListener(OnPermissionGranted listener) {
        mListener = listener;
        return this;
    }

    public PermissionHelper setRationaleListener(RationaleListener listener) {
        mRationaleListener = listener;
        return this;
    }

    public PermissionHelper requestPermissions(String... permissions) {
        if (mListener == null) {
            return this;
        }
        if (hasPermissions(permissions)) {
            mListener.onPermissionsGranted();
        } else if (shouldShowRationale(permissions) && mRationaleListener != null) {
            mRationaleListener.onShowRationale(permissions);
        } else {
            showPermissionsDialog(permissions);
        }
        return this;
    }

    private boolean hasPermissions(String[] permissions) {
        for (String permission : permissions) {
            if (!hasPermission(permission)) {
                return false;
            }
        }
        return true;
    }

    private boolean hasPermission(String permission) {
        return ActivityCompat
                .checkSelfPermission(mContextOperator.getContext(), permission) == PackageManager.PERMISSION_GRANTED;
    }

    private boolean shouldShowRationale(String[] permissions) {
        for (String permission : permissions) {
            if (!hasPermission(permission)) {
                return false;
            }
        }
        return true;
    }

    private boolean shouldShowRationale(String permission) {
        if(mContextOperator instanceof ToolsFragment) {
            return FragmentCompat.shouldShowRequestPermissionRationale((Fragment) mContextOperator, permission);
        } else {
            return ActivityCompat.shouldShowRequestPermissionRationale((Activity) mContextOperator, permission);
        }
    }

    private void showPermissionsDialog(String... permissions) {
        if(mContextOperator instanceof ToolsFragment) {
            FragmentCompat.requestPermissions((Fragment) mContextOperator, permissions, PERMISSIONS_REQUEST_CODE);
        } else {
            ActivityCompat.requestPermissions((Activity) mContextOperator, permissions, PERMISSIONS_REQUEST_CODE);
        }
    }

    @Override
    public void onPermissionsResultPropagated(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == PERMISSIONS_REQUEST_CODE) {
            for (int result : grantResults) {
                if (result != PackageManager.PERMISSION_GRANTED) {
                    mListener.onPermissionsDenied();
                    return;
                }
            }
            mListener.onPermissionsGranted();
        }
    }


    public interface OnPermissionGranted {
        void onPermissionsGranted();

        void onPermissionsDenied();
    }

    public interface RationaleListener {
        void onShowRationale(@NonNull String[] permissions);
    }
}
